<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\EGiftIndividualOrderLine;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitVirtualIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitVirtualIndividualResponse;
use Ignite\Vendor\Hawk\VirtualIndividualOrderLine;
use Ignite\Vendor\Hawk\VirtualRecipient;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class Virtual extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitVirtualIndividual($payload, $requestId);
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        return $this->buildVirtualIndividualRequest();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return SubmitVirtualIndividualResponse::class;
    }

    /**
     * Build the request for the Virtual Individual.
     *
     * @return SubmitVirtualIndividualRequest
     */
    protected function buildVirtualIndividualRequest()
    {
        return SubmitVirtualIndividualRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines(),
            'emailContent' => app(EmailContent::class, [
                'senderName' => config('catalog.vendors.hawk.virtual.senderName'),
                'senderEmail' => config('catalog.vendors.hawk.virtual.senderEmail'),
                'subject' => config('catalog.vendors.hawk.virtual.subject'),
                'emailMessage' => config('catalog.vendors.hawk.virtual.emailMessage'),
                'emailClosingName' => config('catalog.vendors.hawk.virtual.emailClosingName'),
            ]),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return VirtualIndividualOrderLine
     */
    protected function buildOrderLine()
    {
        return VirtualIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int) $this->orderItem->cost,
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return VirtualRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return VirtualRecipient::fromAssociativeArray([
            'id' => $this->order->user_id,
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
        ]);
    }
}
