<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\VendorRepository;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Catalog\Http\Forms\VendorForm;
use Ignite\Catalog\Http\Requests\VendorDeleteManyRequest;
use Ignite\Catalog\Models\Grid\VendorsOrderItemTable;
use Ignite\Catalog\Models\Grid\VendorsTable;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class VendorsController extends Controller
{
    /**
     * @var VendorRepository
     */
    protected $vendorRepository;

    /**
     * Create a new controller instance.
     *
     * @param VendorRepository $vendorRepository
     */
    public function __construct(VendorRepository $vendorRepository)
    {
        $this->vendorRepository = $vendorRepository;
    }

    /**
     * Display all vendors.
     *
     * @param  VendorsTable  $table
     * @param  Request  $request
     * @param  string  $status
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(VendorsTable $table, Request $request, $status = 'active')
    {
        $status = strtolower($status);
        $table->with('status', $status);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.vendors.index', compact('status'));
    }

    /**
     * Show a vendor.
     *
     * @param  Vendor  $vendor
     * @param  VendorsOrderItemTable  $table
     * @param  Request  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function show(Vendor $vendor, VendorsOrderItemTable $table, Request $request)
    {
        $table->with('vendor', $vendor);

        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.vendors.show', compact('vendor'));
    }

    /**
     * Show the create form for a new catalog.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(VendorForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.vendors.create')
        ]);

        return view('Catalog::admin.vendors.create', compact('form'));
    }

    /**
     * Store a new catalog.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function store(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(VendorForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $vendor = $this->vendorRepository->create(collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::vendor.store.success', ['name' => $vendor->name]);
            return redirect()->route('admin.catalog.vendors.edit', $vendor);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::vendor.store.failure');
            return redirect()->route('admin.catalog.vendors.index');
        }
    }

    /**
     * Show the edit form for an existing catalog.
     *
     * @param  Vendor      $vendor
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function edit(Vendor $vendor, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(VendorForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.vendors.update', $vendor),
            'model' => $vendor
        ]);

        return view('Catalog::admin.vendors.edit', compact('form', 'vendor'));
    }

    /**
     * Update an existing catalog.
     *
     * @param  int $vendor
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function update($vendor, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(VendorForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $vendor = $this->vendorRepository->update($vendor, collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::vendor.update.success', ['name' => $vendor->name]);
            return redirect()->route('admin.catalog.vendors.edit', $vendor);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::vendor.update.failure');
            return redirect()->route('admin.catalog.vendors.index');
        }
    }

    /**
     * Delete a vendor.
     *
     * @param  int $vendor
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($vendor)
    {
        try {
            $vendor = $this->vendorRepository->delete($vendor);
            $status = 'success';
            $message = trans('Catalog::vendor.delete.success', ['name' => $vendor->name]);
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = trans('Catalog::vendor.delete.failure');
        }

        if (request()->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.catalog.vendors.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  VendorDeleteManyRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(VendorDeleteManyRequest $request)
    {
        $ids = $request->ids;
        $vendors = explode(',', $ids);

        try {
            $this->vendorRepository->deleteMany($vendors);
            $this->flashSuccess('Catalog::vendor.delete_many.success', compact('ids'));
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::vendor.delete_many.failure');
        }

        return redirect()->route('admin.catalog.vendors.index');
    }
}
