<?php

namespace Ignite\Catalog\Console;

use Ignite\Catalog\Console\CatalogItemImportCommandAbstract;
use Ignite\Catalog\Entities\Vendor;

class TangoImport extends CatalogItemImportCommandAbstract
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:tango-import';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import Tango catalog items with an xlsx file.';

    /**
     * @param  string $filename
     * @return array
     */
    protected function getData(string $filename): array
    {
        return $this->getExcelData($filename);
    }

    /**
     * {@inheritdoc}
     */
    protected function process(array $data)
    {
        $data = $this->groupBySku($data);
        return parent::process($data);
    }

    /**
     * @param  array $rows
     * @return void
     */
    protected function introMessage(array $rows)
    {
        parent::introMessage($rows);
        $this->warn(
            'Items with the same SKU will be grouped together as different denominations.'
            . 'We will only use the description, disclaimers, terms, etc. from the first item in the group.'
        );
    }

    /**
     * Group the rows by SKU and take only the first one with the same SKU,
     * but combine the denominations or prices.
     *
     * @param  array  $data
     * @return array
     */
    protected function groupBySku(array $data): array
    {
        $rowsbySku = [];
        $denominationsBySku = [];

        // for each sku, group the rows and denominations
        foreach ($data as $number => $row) {
            $sku = $row['sku'];

            $denominations = trim($row['denominations']);
            if (is_string($denominations) && !empty($denominations)) {
                $denominations = explode(',', $denominations);
            }

            if (is_array($denominations) && count($denominations) >= 1) {
                foreach ($denominations as $denominationString) {
                    list($label, $amount) = $this->splitDenomination($denominationString);
                    $this->checkExpectedDenomination(
                        $row['name'],
                        $amount,
                        $row['cost_usd'],
                        $row['exchange_to_usd'] ?? 1
                    );
                    $denominationsBySku[$sku][$denominationString] = $row['utid'];
                }
            } elseif (!is_array($denominations) || count($denominations) == 0) {
                $denomination = floor($this->exchangeFromUsd(
                    $row['cost_usd'] ?: $row['msrp_usd'],
                    $row['exchange_to_usd']
                ));
                $denominationsBySku[$sku][$denomination] = $row['utid'];
            }
            if ($row['price_usd']) {
                $this->checkExpectedPrice($row['name'], $row['price_usd'], $row['cost_usd'], $row['markup']);
                $this->checkExpectedPoints($row['name'], $row['points'], $row['price_usd']);
            }

            $rowsbySku[$sku][] = $row;
        }

        // for each sku, create one row
        $skus = [];
        foreach ($rowsbySku as $sku => $rows) {
            $row = $rows[0];
            $denominations = $denominationsBySku[$sku];
            ksort($denominations);

            $row['denominations'] = implode(',', array_keys($denominations));
            $row['utid'] = json_encode($denominationsBySku[$sku]);

            // use the first row only for all other fields
            if (count($rows) > 1) {
                $row['msrp_usd'] = null;
                $row['cost_usd'] = null;
                $row['price_usd'] = null;
                $row['points'] = null;
            }
            $skus[] = $row;
        }

        return $skus;
    }

    /**
     * {@inheritdoc}
     */
    protected function prepareRow(int $number, array $row): array
    {
        $row['class'] = 'egift';
        $row['image'] = basename($row['image']);

        $renames = [
            'msrp_usd' => 'msrp',
            'cost_usd' => 'cost',
            'price_usd' => 'price',
            'markup' => 'price_markup',
        ];
        foreach ($renames as $from => $to) {
            if (isset($row[$from])) {
                $row[$to] = $row[$from];
                unset($row[$from]);
            }
        }

        return $row;
    }

    /**
     * {@inheritdoc}
     */
    protected function createItemFromDenomination(string $denominationString, array $row)
    {
        $utids = json_decode($row['utid'], true);
        $row['vendor_meta']['account_identifier'] = $row['account_identifier'];
        $row['vendor_meta']['campaign'] = $row['campaign'] ?? null;
        $row['vendor_meta']['etid'] = $row['etid'];
        $row['vendor_meta']['utid'] = $utids[$denominationString];

        return parent::createItemFromDenomination($denominationString, $row);
    }

    /**
     * {@inheritdoc}
     */
    protected function vendor()
    {
        if (is_null($this->vendor)) {
            $this->vendor = Vendor::where('name', 'tango')->first();
            if (is_null($this->vendor)) {
                throw new \Exception('No vendors exist. Run `module:seed` first.');
            }
        }

        return $this->vendor;
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        $options = parent::getOptions();
        $options['file'] = [
            $options['file'][0],
            $options['file'][1],
            $options['file'][2],
            $options['file'][3],
            'vendor/ignite/modules/Catalog/src/Console/data/tango/egifts.xlsx',
        ];
        return $options;
    }
}
