<?php

namespace Ignite\Catalog\Models\Report;

use Ignite\Catalog\Models\Grid\OrderItemFormatter;
use Ignite\Core\Entities\User;
use Ignite\Core\Models\Grid\QueryTable;
use Illuminate\Database\DatabaseManager;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class OrderItemReport extends QueryTable
{
    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'username', 'company'
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'user_id'
    ];

    /**
     * @var OrderItemFormatter
     */
    protected $formatter;

    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        OrderItemFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Order Date',
                'name' => 'catalog_order.created_at',
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'email' => [
                'title' => 'E-mail',
                'name' => 'participant.email',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_email' => [
                'title' => 'Recipient Email',
                'name' => 'catalog_order.ship_email',
                'orderable' => true,
                'exportable' => true,
            ],
            'number' => [
                'title' => 'Order #',
                'name' => 'catalog_order.number',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor_order_number' => [
                'title' => 'Vendor Order #',
                'name' => 'catalog_order_item.vendor_order_number',
                'orderable' => true,
                'exportable' => true,
            ],
            'name' => [
                'title' => 'Item Name',
                'name' => 'catalog_order_item.name',
                'orderable' => true,
                'exportable' => true,
            ],
            'class' => [
                'title' => 'Item Type',
                'name' => 'catalog_order_item.class',
                'orderable' => true,
                'exportable' => true,
            ],
            'category' => [
                'title' => 'Category',
                'name' => 'catalog_order_item.category',
                'orderable' => true,
                'exportable' => true,
            ],
            'points' => [
                'title' => 'Points',
                'name' => 'catalog_order_item.points',
                'orderable' => true,
                'exportable' => true,
            ],
            'price' => [
                'title' => 'Price',
                'name' => 'catalog_order_item.price',
                'orderable' => true,
                'exportable' => true,
            ],
            'quantity' => [
                'title' => 'Quantity',
                'name' => 'catalog_order_item.quantity',
                'orderable' => true,
                'exportable' => true,
            ],
            'total' => [
                'title' => 'Total',
                'name' => 'catalog_order_item_derived.total',
                'orderable' => true,
                'exportable' => true,
                'searchable' => true,
            ],
            // 'total_dollar' => [
            //     'title' => 'Total Dollar',
            //     'name' => 'catalog_order_item_derived.total_dollar',
            //     'orderable' => true,
            //     'exportable' => true,
            //     'searchable' => true,
            // ],
            'processed' => [
                'title' => 'Processed',
                'name' => 'catalog_order_item.processed',
                'orderable' => true,
                'exportable' => true,
            ]
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = $this->getColumnNames()->toArray();

        $pointValue = config('catalog.default_point_value');
        $orderItemSubquery = "(SELECT id, (points * quantity) as total,"
            . " (points * quantity * {$pointValue}) as total_dollar FROM catalog_order_item)"
            . " AS catalog_order_item_derived";

        return $this->getConnection()
            ->table('catalog_order_item')
            ->select($columns)
            ->leftJoin('catalog_order', 'catalog_order.id', '=', 'catalog_order_item.catalog_order_id')
            ->leftJoin('core_participant as participant', 'participant.user_id', '=', 'catalog_order.user_id')
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->leftJoin(DB::raw($orderItemSubquery), 'catalog_order_item.id', '=', 'catalog_order_item_derived.id')
            ->where('user.internal', User::TYPE_PARTICIPANT);
    }

    /**
     * Get default builder parameters.
     *
     * @param array $params
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        $columns = array_flip(array_keys($this->getColumns()));

        $params = parent::getBuilderParameters();
        $params['order'] = [
            [$columns['created_at'], 'desc'],
        ];

        return $params;
    }

    /**
     * @inheritdoc
     */
    protected function getColumnFormattingMap()
    {
        return [
            'points' => [$this->formatter, 'points'],
            'total' => [$this->formatter, 'total'],
            'processed' => [$this->formatter, 'processed']
        ];
    }
}
