<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Core\Entities\Participant;
use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\FundingOrderLine;
use Ignite\Vendor\Hawk\PersonalizedOrderLine;
use Ignite\Vendor\Hawk\PersonalizedRecipient;
use Ignite\Vendor\Hawk\Recipient;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\ShippingMethod;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class Reloadable extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var Participant
     */
    protected $participant = null;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $this->createCardOrReloadCard($payload, $requestId, $hawkApi);
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        if ($this->hasCard()) {
            return $this->buildSubmitFundingRequest();
        }

        return $this->buildOpenLoopPersonalizedIndividualRequest();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return [
            SubmitOpenLoopPersonalizedIndividualResponse::class,
            SubmitFundingResponse::class
        ];
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitOpenLoopPersonalizedIndividualRequest
     */
    protected function buildOpenLoopPersonalizedIndividualRequest()
    {
        return SubmitOpenLoopPersonalizedIndividualRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines(),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return PersonalizedOrderLine
     */
    protected function buildOrderLine()
    {
        return PersonalizedOrderLine::fromAssociativeArray([
            'amount' => $this->getAmount(),
            'description' => config('catalog.vendors.hawk.reloadable.description', ''),
            'fourthLineEmbossText' => config('catalog.vendors.hawk.reloadable.fourth_line_emboss', 'THANK YOU'),
            'cardCarrierMessage' => config('catalog.vendors.hawk.reloadable.card_carrier_message', ''),
            'shippingMethod' => config('catalog.vendors.hawk.reloadable.shipping_method', ShippingMethod::USPS_FIRST_CLASS),
            'recipient' => $this->getRecipientData(),
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return PersonalizedRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        $recipientId = $this->getRecipientId();
        if (empty($recipientId)) {
            $recipientId = $this->generateRecipientId();
        }

        return PersonalizedRecipient::fromAssociativeArray([
            'id' => $recipientId,
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->getCountry(),
            ])
        ]);
    }

    /**
     * Create a card or reload it.
     *
     * @param $payload
     * @param $requestId
     * @param Client $hawkApi
     *
     * @return ErrorResponse|SubmitFundingResponse|SubmitOpenLoopPersonalizedIndividualResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function createCardOrReloadCard($payload, $requestId, Client $hawkApi)
    {
        if ($this->hasCard()) {
            return $hawkApi->submitFunding($payload, $requestId);
        }

        $response = $hawkApi->submitOpenLoopPersonalizedIndividual($payload, $requestId);

        if (empty($this->getRecipientId())) {
            $payloadInfo = $payload->toArray();
            $this->updateRecipientId($payloadInfo['orderDetails'][0]['recipient']['id']);
        }

        return $response;
    }

    /**
     * Build the submit funding request.
     *
     * @return SubmitFundingRequest
     */
    protected function buildSubmitFundingRequest()
    {
        return SubmitFundingRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => [$this->buildFundingLines()],
        ]);
    }

    /**
     * Build the FundingOrderLine.
     *
     * @return FundingOrderLine
     */
    protected function buildFundingLines()
    {
        return FundingOrderLine::fromAssociativeArray([
            'amount' => $this->getAmount(),
            'recipient' => Recipient::fromAssociativeArray([
                'id' => $this->getRecipientId(),
            ])
        ]);
    }

    /**
     * Determine whether the participant has a reloadable card already registered.
     *
     * @return bool
     */
    protected function hasCard(): bool
    {
        return !empty($this->getRecipientId());
    }

    /**
     * Get the participant reloadable recipient id.
     *
     * @return string
     */
    protected function generateRecipientId()
    {
        $this->guardAgainstMissingParticipant();

        // @todo: we can check for a ReloadableParticipantTrait or ReloadableParticipantInterface
        //        for use of a custom generateRecipientId()
        // return $this->order->participant->generateRecipientId();

        $prefix = config('catalog.vendors.hawk.reloadable.recipient_id_prefix');
        $identityColumn = config('catalog.vendors.hawk.reloadable.identity_column');

        if (strlen($prefix) < 3) {
            throw new \InvalidArgumentException("Invalid recipient_id_prefix, requires minimum length of 3 character(s): $prefix");
        }

        return $prefix . $this->order->participant->getAttribute($identityColumn);
    }

    /**
     * Get the participant Hawk recipient id.
     *
     * @return null|string
     */
    protected function getRecipientId(): ?string
    {
        $this->guardAgainstMissingParticipant();

        // These orders may be on a queue, and may be executed in parallel or
        // out of order, or with delay. So in case another order is made before
        // this one, we first check with the database to see if there is a
        // reloadable_id saved now.
        if (is_null($this->participant)) {
            $this->participant = $this->order->participant;
            $this->participant->refresh();
        }

        return $this->participant->reloadable_id;
    }

    /**
     * Update the participant Hawk recipient id.
     *
     * @param  string $recipientId
     * @return self
     */
    protected function updateRecipientId(string $recipientId)
    {
        $this->guardAgainstMissingParticipant();

        $this->order->participant->setAttribute('reloadable_created_at', now()->format('Y-m-d H:i:s'));
        $this->order->participant->setAttribute('reloadable_id', $recipientId);
        $this->order->participant->save();

        return $this;
    }

    /**
     * Gets the dollar amount the card is for.
     *
     * @return int
     */
    protected function getAmount(): int
    {
        if (isset($this->orderItem->vendor_meta['reloadable_amount']) && $this->orderItem->vendor_meta['reloadable_amount'] > 0) {
            // the amount we send may be different from the "price"
            return floor($this->orderItem->vendor_meta['reloadable_amount']);
        } elseif ($this->orderItem->price > 0) {
            return floor($this->orderItem->price);
        } elseif ($this->orderItem->cost > 0) {
            ray(3, $this->orderItem->cost);
            return floor($this->orderItem->cost);
        } elseif ($this->orderItem->points > 0 && $this->orderItem->point_value > 0) {
            return floor($this->orderItem->points * $this->orderItem->point_value);
        }
        // else - do not use the points amount alone!

        throw new \InvalidArgumentException('Can not determine the dollar amount to send.');
    }

    /**
     * Guard that we have a participant loaded.
     *
     * @throws InvalidArgumentException
     */
    protected function guardAgainstMissingParticipant()
    {
        if (! $this->order->user_id) {
            throw new \InvalidArgumentException('No user associated with the order.');
        }

        if (! $this->order->relationLoaded('participant')) {
            $this->order->load('participant');
        }

        if (! $this->order->participant) {
            throw new \InvalidArgumentException('Unable to load the participant associated with the order.');
        }
    }
}
