<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\ClosedLoopIndividualOrderLine;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class Giftcard extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitClosedLoopIndividual($payload, $requestId);
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        return $this->buildClosedLoopIndividualRequest();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return SubmitClosedLoopIndividualResponse::class;
    }

    /**
     * Build the request for the giftcard Individual.
     *
     * @return SubmitClosedLoopIndividualRequest
     */
    protected function buildClosedLoopIndividualRequest()
    {
        return SubmitClosedLoopIndividualRequest::fromAssociativeArray([
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'clientProgramNumber' => $this->getProductId(),
            'orderDetails' => $this->buildIndividualOrderLines(),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildIndividualOrderLines()
    {
        return array_map(function () {
            return $this->buildIndividualOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return ClosedLoopIndividualOrderLine
     */
    protected function buildIndividualOrderLine()
    {
        if (empty($this->orderItem->item->vendor_meta['content_provider_code'])) {
            throw new \DomainException("Missing content provider code for Item ID: {$this->orderItem->item->id}");
        }

        return ClosedLoopIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int) $this->orderItem->cost,
            'quantity' => $this->orderItem->quantity,
            'contentProvider' => $this->orderItem->item->vendor_meta['content_provider_code'],
            'groupingId' => $this->order->number
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return array
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return [
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'id' => $this->order->user_id,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->getCountry(),
            ])
        ];
    }
}
