<?php

namespace Ignite\Catalog\Tests\Unit\Models;

use DomainException;
use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Tests\TestCase;

class CartTest extends TestCase
{
    public function setUp() : void
    {
        parent::setUp();
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_create_a_cart_model_instance()
    {
        $this->assertInstanceOf(Cart::class, new Cart(collect([]), 0.00, 0.00));
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_return_the_items_it_was_constructed_with()
    {
        $items = collect(['a', 'b', 'c']);

        $cart = new Cart($items, 0.00, 0.00);

        $this->assertSame($items, $cart->items());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_return_the_balance_it_was_constructed_with()
    {
        $items = collect(['a', 'b', 'c']);

        $cart = new Cart($items, 0.00, 0.00);

        $this->assertSame('0', $cart->balance());
    }

    /**
 * @test
 * @group Catalog
 * @group Cart
 */
    public function it_can_return_the_total_it_was_constructed_with()
    {
        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, 300, 300);

        $this->assertEquals(300, $cart->total());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_will_throw_an_exception_when_the_calculated_total_is_different_to_the_value_passed_through_the_constructor()
    {
        $this->expectException(DomainException::class);

        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, 300, 0);

        $cart->total();
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_cart_is_empty()
    {
        $items = collect([]);

        $cart = new Cart($items, 0.00, 0.00);

        $this->assertTrue($cart->empty());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_cart_is_not_empty()
    {
        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, 0.00, 0.00);

        $this->assertTrue($cart->notEmpty());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_get_the_total_quantity()
    {
        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, 0.00, 300);

        $this->assertSame(3, $cart->quantity());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_calculate_the_total_deficit()
    {
        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, 600, 300);

        $this->assertSame('300', $cart->deficit());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_cart_can_be_checked_out()
    {
        $items = collect([
            tap(new CartItem())->forceFill(['quantity' => 2, 'points' => 100]),
            tap(new CartItem())->forceFill(['quantity' => 1, 'points' => 100])
        ]);

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertSame(true, $cart->canCheckout());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_order_contains_a_giftcard()
    {
        $items = $this->createCartItemCollection(1, [
            'item' => [
                'class' => 'giftcard'
            ]
        ]);

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertTrue($cart->hasGiftCard());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_order_contains_a_physical_item()
    {
        $items = $this->createCartItemCollection(1, [
            'item' => [
                'class' => 'giftcard'
            ]
        ]);

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertTrue($cart->hasPhysical());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_order_contains_a_non_physical_item()
    {
        $items = $this->createCartItemCollection(1, [
            'item' => [
                'class' => 'egift'
            ]
        ]);

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertTrue($cart->hasNonPhysical());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_order_contains_only_physical_items()
    {
        $items = $this->createCartItemCollection(2, [
            'item' => [
                'class' => 'merchandise'
            ]
        ]);

        $items->push($this->createCartItem([
            'item' => [
                'class' => 'merchandise'
            ]
        ]));

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertTrue($cart->hasOnlyPhysical());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_determine_whether_the_order_contains_only_non_physical_items()
    {
        $items = $this->createCartItemCollection(2, [
            'item' => [
                'class' => 'egift'
            ]
        ]);

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertTrue($cart->hasOnlyNonPhysical());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_count_physical_items()
    {
        $items = $this->createCartItemCollection(2, [
            'item' => [
                'class' => 'egift'
            ]
        ]);

        $items->push($this->createCartItem([
            'item' => [
                'class' => 'merchandise'
            ]
        ]));

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertEquals(1, $cart->countPhysical());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     */
    public function it_can_count_non_physical_items()
    {
        $items = $this->createCartItemCollection(2, [
            'item' => [
                'class' => 'egift'
            ]
        ]);

        $items->push($this->createCartItem([
            'item' => [
                'class' => 'merchandise'
            ]
        ]));

        $cart = new Cart($items, $balance = 600, $total = 300);

        $this->assertEquals(2, $cart->countNonPhysical());
    }

    /**
     * Create a simple collection of cart items.
     *
     * @param int   $howMany
     * @param array $overrides
     * @return \Illuminate\Support\Collection
     */
    private function createCartItemCollection($howMany = 1, $overrides = [])
    {
        $collection = collect();

        foreach (range(1, $howMany) as $number) {
            $collection->push($this->createCartItem($overrides));
        }

        return $collection;
    }

    /**
     * Create a cart item.
     *
     * @param  array $overrides
     * @return CartItem
     */
    private function createCartItem($overrides = [])
    {
        $data = array_merge([
            'catalog_id' => 2,
            'catalog_vendor_id' => 2,
            'item_id' => 10001,
            'quantity' => 1,
            'price' => 100,
            'points' => 100
        ], $overrides);

        if (array_key_exists('item', $data)) {
            $item = tap(new Item())->forceFill($data['item']);
            unset($data['item']);
        }

        /** @var CartItem $cartItem */
        $cartItem = tap(new CartItem())->forceFill($data);

        if (isset($item)) {
            $cartItem->setRelation('item', $item);
        }

        return $cartItem;
    }
}
