<?php

namespace Ignite\Catalog\Listeners;

use Ignite\Catalog\Events\IncompleteTangoOrder;
use Ignite\Catalog\Jobs\Tango\ProcessManager;
use Ignite\Vendor\Tango\Client;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;

class RequestTangoOrderInfo implements ShouldQueue
{
    use InteractsWithQueue;

    /**
     * The time (seconds) before the job should be processed. E.g. 300 = 5 minutes.
     *
     * @var int
     */
    public $delay = 300;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ProcessManager
     */
    protected $manager;

    /**
     * RequestTangoOrderInfo constructor.
     *
     * @param Client $client
     */
    public function __construct(Client $client)
    {
        $this->client = $client;
    }

    /**
     * Handle the event.
     *
     * @param IncompleteTangoOrder $event
     *
     * @return bool
     */
    public function handle(IncompleteTangoOrder $event): bool
    {
        $orderNumber = $event->orderItem->vendor_order_number ?? null;

        // fallback to vendor_meta column for any older ignite programs.
        if (!$orderNumber) {
            $attempts = array_reverse($event->orderItem->vendor_meta['attempts']);
            foreach ($attempts as $attempt) {
                if (empty($attempt['response']['referenceOrderID'])) {
                    continue;
                }

                $orderNumber = $attempt['response']['referenceOrderID'];
                break;
            }
        }

        // fallback to vendor_meta column for any older ignite programs.
        $requestId = $event->requestId ?? key($event->orderItem->vendor_meta['attempts']);
        $this->processOrderInfo($event, $orderNumber, $requestId);

        $this->delete();

        return false;
    }

    /**
     * @param  $event
     * @param  $orderNumber
     * @param  $requestId
     * @return bool
     */
    protected function processOrderInfo($event, $orderNumber, $requestId): bool
    {
        return true;

        // @todo: we make a Tango call to check on new status of the order and update if needed
        $orderInfoResponse = $this->client->orderInfo($orderNumber, $requestId);
        $manager = $this->getProcessManager($event, $requestId);

        if ($orderInfoResponse->getStatusCode() == 200) {
            return $manager->complete($orderInfoResponse);
        }
    }

    /**
     * Create an instance of a process manager.
     *
     * @param IncompleteTangoOrder $event
     * @param string              $requestId
     * @return ProcessManager
     */
    protected function getProcessManager(IncompleteTangoOrder $event, $requestId)
    {
        if (is_null($this->manager)) {
            return new ProcessManager($event->order, $event->orderItem, $requestId);
        }

        return $this->manager;
    }
}
