<?php

namespace Ignite\Catalog\Console;

use Ignite\Catalog\Console\CatalogItemImportCommandAbstract;
use Ignite\Catalog\Entities\Vendor;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

/**
 * Sample json for items with price:
 *   {
 *     "sku" : "amazon_egift_us",
 *     "catalog" : "gift-cards-us",
 *     "type": "simple",
 *     "class" : "egift",
 *     "locale" : "en",
 *     "name" : "Amazon.com eGift",
 *     "short_description" : null,
 *     "description" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "terms" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "manufacturer" : "Amazon.com",
 *     "image" : "amazon.png",
 *     "msrp" : 0,
 *     "cost" : 0,
 *     "price": 100,
 *     "price_markup": 0.15,
 *     "point_value" : 0.005,
 *     "points" : 23000,
 *     "visibility" : "1",
 *     "active" : "1",
 *     "vendor_active": 1,
 *     "vendor_meta" : {"content_provider_code": "Amazon_eGift"},
 *     "categories" : "General|Home & Garden"
 *   },
 *
 * Sample json for items with denominations:
 *   {
 *     "sku" : "amazon_egift_us",
 *     "catalog" : "gift-cards-us",
 *     "type": "configurable",
 *     "class" : "egift",
 *     "locale" : "en",
 *     "name" : "Amazon.com eGift",
 *     "short_description" : null,
 *     "description" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "terms" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "manufacturer" : "Amazon.com",
 *     "image" : "amazon.png",
 *     "denominations" : "25,50,100,500",
 *     "price_markup": 0.15,
 *     "point_value" : 0.005,
 *     "visibility" : "1",
 *     "active" : "1",
 *     "vendor_active": 1,
 *     "vendor_meta" : {"content_provider_code": "Amazon_eGift"},
 *     "categories" : "General|Home & Garden"
 *   },
 *
 * Note: you can use unix commands like 'jq' or 'csvtojson'/'json2csv' to convert .csv files to json format and reverse.
 *
 * @todo: update sample file Catalog\src\Console\data\hawk\egift.json and Catalog\src\Console\data\hawk\giftcard.json
 */

class HawkImport extends CatalogItemImportCommandAbstract
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:hawk-import';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import Hawk catalog data into Ignite using a json file.';

    /**
     * @var string
     */
    protected $product;

    /**
     * @var string
     */
    protected $denominations;

    /**
     * @param  string $filename
     * @return array
     */
    protected function getData(string $filename): array
    {
        return $this->getJsonData($filename);
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $this->product = $this->argument('product');
        $this->denominations = $this->option('denominations');

        return parent::handle();
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            ['product', InputArgument::REQUIRED, 'The product ID for this program.'],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        $fileDescription = 'The file from which to import card data, by default in the storage folder.';

        return [
            ['file', 'f', InputOption::VALUE_OPTIONAL, $fileDescription, storage_path('giftcards.csv')],
            ['denominations', null, InputOption::VALUE_OPTIONAL, 'The denominations to create for each card.', null],
            ['reset', 'r', InputOption::VALUE_NONE, 'Truncates catalog related tables; not available on production.'],
            // ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * {@inheritdoc}
     */
    protected function prepareRow(int $number, array $row): array
    {
        $row['class'] = 'egift';
        $row['image'] = basename($row['image']);
        $row['denominations'] = $row['denominations'] ?? $this->denominations;
        $row['vendor_meta']['product_id'] = $this->product;

        return $row;
    }

    /**
     * The catalog vendor entity.
     *
     * @return Vendor
     * @throws \Exception
     */
    protected function vendor()
    {
        if (is_null($this->vendor)) {
            $this->vendor = Vendor::where('name', 'hawk')->first();
            if (is_null($this->vendor)) {
                throw new \Exception('No vendors exist. Run `module:seed` first.');
            }
        }

        return $this->vendor;
    }
}
