<?php

namespace Ignite\Catalog\Models;

use Ignite\Core\Entities\Participant;

class ReloadableThreshold
{
    /**
     * An instance of the Cart model.
     *
     * @var Cart
     */
    protected $cart;

    /**
     * The date the participant created the reloadable card.
     *
     * @var null|string|bool
     */
    protected $reloadable_created_at = false;

    /**
     * Cache the check value for whether they have ordered previously.
     *
     * @var bool
     */
    protected $isFirstOrder;

    /**
     * ReloadableVerification constructor.
     *
     * @param Cart $cart
     */
    public function __construct(Cart $cart)
    {
        $this->cart = $cart;
    }

    /**
     * Verify whether the participant can add funds to their reloadable card.
     *
     * @return bool
     */
    public function verify(): bool
    {
        return $this->hasCreatedReloadableCard() || $this->hasNotCreatedCardButHasEnoughPoints();
    }

    /**
     * The threshold of amount the participant can reach.
     *
     * @return int
     */
    public function amount(): int
    {
        $userBalance = (int) str_replace(',', '', auth()->user()->points());

        return floor($userBalance * config('catalog.default_point_value'));
    }

    /**
     * Determine whether this would be the user's first reloadable order.
     *
     * @return bool
     */
    public function isFirstOrder(): bool
    {
        if ($this->isFirstOrder === null) {
            $reloadableId = Participant::query()
                ->where('user_id', auth()->id())
                ->value('reloadable_id');
            $this->isFirstOrder = !$reloadableId;
        }

        return $this->isFirstOrder;
    }

    /**
     * The date the reloadable card was created by the participant.
     *
     * @return bool|null|string
     */
    public function getReloadableCreatedAt()
    {
        if (false === $this->reloadable_created_at) {
            $participant = Participant::query()->where('user_id', auth()->id())->first();
            $this->reloadable_created_at = $participant->reloadable_created_at;
        }

        return $this->reloadable_created_at;
    }

    /**
     * Determine whether the participant has created the reloadable card.
     *
     * @return bool
     */
    public function hasCreatedReloadableCard(): bool
    {
        return ! $this->isFirstOrder();
    }

    /**
     * Determine whether the participant has not created the reloadable card
     * yet but now has enough points to do so.
     *
     * @return bool
     */
    public function hasNotCreatedCardButHasEnoughPoints(): bool
    {
        return (! $this->hasCreatedReloadableCard()) && ($this->hasEnoughPoints());
    }

    /**
     * Determine whether the participant has enough points to create the reloadable card.
     *
     * @return bool
     */
    public function hasEnoughPoints(): bool
    {
        $points = (int) str_replace([',', '.'], '', $this->cart->balance());

        return (bool) $points >= $this->amount();
    }

    /**
     * Get the minimum required amount to create the reloadable card.
     *
     * @return int
     */
    public function getMinRequiredAmount(): int
    {
        if ($this->hasCreatedReloadableCard()) {
            $min = config('catalog.vendors.hawk.reloadable.min_threshold_reload', 0);
        } else {
            $min = config('catalog.vendors.hawk.reloadable.min_threshold_initial', 0);
        }

        return max($min, 1);
    }

    /**
     * The reloadable card item sku.
     *
     * @return string
     */
    public function sku(): string
    {
        return config('catalog.vendors.hawk.reloadable.sku') ?? 'visa_reloadable';
    }
}
