<?php

namespace Ignite\Catalog\Jobs;

use Ignite\Catalog\Emails\OrderItemFailure;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Facades\Mail;

abstract class ProcessManager
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * @var string
     */
    protected $requestId;

    /**
     * ProcessManager constructor.
     *
     * @param Order $order
     * @param OrderItem $orderItem
     * @param $requestId
     */
    public function __construct(Order $order, OrderItem $orderItem, $requestId)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
        $this->requestId = $requestId;
    }

    /**
     * Complete the order.
     *
     * @param $response
     * @param array $meta
     *
     * @return bool
     */
    public function complete($response, array $meta = [])
    {
        $this->process($response, [
            'processed' => 1,
            'processed_at' => $now = now(),
            'processed_quantity' => $qty = $this->order->totalQuantity(),
            'shipped' => 1,
            'shipped_at' => $now,
            'shipped_quantity' => $qty,
            'cancelled' => 0,
            'cancelled_at' => null,
            'cancelled_quantity' => 0,
        ], $meta);

        $this->order->refresh();

        if ($this->order->hasProcessedAllItems()) {
            $this->order->update([
                'processed' => 1,
                'processed_at' => $now,
                'cancelled' => 0,
                'cancelled_at' => null,
            ]);
        }

        return true;
    }

    /**
     * Cancel the order.
     *
     * @param $response
     * @param array $meta
     *
     * @return bool
     */
    public function cancel($response, array $meta = [])
    {
        $this->process($response, [
            'processed' => 0,
            'processed_at' => null,
            'processed_quantity' => 0,
            'shipped' => 0,
            'shipped_at' => null,
            'shipped_quantity' => 0,
            'cancelled' => 1,
            'cancelled_at' => $now = now(),
            'cancelled_quantity' => $this->order->totalQuantity(),
        ], $meta);

        if ($this->order->hasCancelledAllItems()) {
            $this->order->update([
                'processed' => 0,
                'processed_at' => null,
                'cancelled' => 1,
                'cancelled_at' => $now,
            ]);
        }

        return true;
    }

    /**
     * Store the attempt.
     *
     * @param $response
     * @param array $meta
     *
     * @return bool
     */
    public function attempt($response, array $meta = [])
    {
        return $this->process($response, [], $meta);
    }

    /**
     * Report the error.
     *
     * @param ErrorResponse $response
     * @param array $meta
     *
     * @return bool
     */
    public function error($response, array $meta = [])
    {
        // We're not cancelling the item yet, instead, we'll send an email with the reason.
        $this->process($response, [], $meta);
        Mail::send(new OrderItemFailure($this->order, $this->orderItem));

        return false;
    }

    /**
     * Process the response.
     *
     * @param $response
     * @param array $data
     * @param array $meta
     *
     * @return bool
     */
    protected function process(Arrayable $response, array $data = [], array $meta = [])
    {
        $existing = $this->orderItem->vendor_meta ?? [];

        if (! isset($existing['attempts'])) {
            $existing['attempts'] = [];
        }

        $existing['attempts'][$this->requestId] = [
            'response' => $response->toArray(),
            'item' => $this->orderItem->item->vendor_meta ?? []
        ];

        if (array_key_exists('attempts', $meta)) {
            unset($meta['attempts']);
        }

        $vendorMeta = array_merge($existing, $meta);
        // if there are multiple categories, just get the first one
        $category = $this->orderItem->item ? $this->orderItem->item->categories()->first() : null;

        $this->orderItem->update(array_merge([
            'vendor_order_number' => $this->getResponseOrderNumber($vendorMeta),
            'category' => $category ? $category->name : null,
            'vendor_meta' => $vendorMeta,
        ], $data));

        return true;
    }

    /**
     * Gets the order number from the response.
     *
     * @param  array  $meta
     * @return null|string
     */
    abstract protected function getResponseOrderNumber(array $meta);

    /**
     * Determine whether the order response is complete.
     *
     * @param BaseOrder$response
     *
     * @return bool
     */
    abstract public function isOrderComplete($response);
}
