<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Ignite\Catalog\Entities\OrderItem;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Vendor\Hawk\Client;

class OrderItemsController extends Controller
{
    /**
     * Inspect the order item details.
     *
     * @param OrderItem $orderItem
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function inspect(OrderItem $orderItem)
    {
        $orderItem = $orderItem->load(['vendor']);
        $orderItem = $this->hasVendorMetaData($orderItem) ? $this->refreshOrderStatus($orderItem) : $orderItem;

        return view('Catalog::admin.orders.items.inspect', compact('orderItem'));
    }

    /**
     * @param  OrderItem $orderItem
     * @return OrderItem
     */
    protected function refreshOrderStatus(OrderItem $orderItem)
    {
        try {
            $orderInfoResponse = null;
            if ($this->isHawkItem($orderItem)) {
                $orderInfoResponse = $this->getHawkOrderInfo($orderItem);
            } elseif ($this->isTangoItem($orderItem)) {
                $orderInfoResponse = $this->getTangoOrderInfo($orderItem);
            }
        } catch (\Exception $e) {
            $orderInfoResponse = [
                'errors' => [
                    [
                        'errorCode' => $e->getCode(),
                        'message' => $e->getMessage(),
                    ],
                ],
            ];
        }

        $orderItem->vendor_meta = array_merge(
            $orderItem->vendor_meta,
            ['inspect' => $orderInfoResponse],
        );

        return $orderItem;
    }

    /**
     * @param  OrderItem $orderItem
     * @return bool
     */
    protected function isHawkItem(OrderItem $orderItem): bool
    {
        if (stripos($orderItem->vendor->name, 'hawk') !== false
            || stripos($orderItem->vendor->redemption_class, 'hawk') !== false
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param  OrderItem $orderItem
     * @return bool
     */
    protected function isTangoItem(OrderItem $orderItem): bool
    {
        if (stripos($orderItem->vendor->name, 'tango') !== false
            || stripos($orderItem->vendor->redemption_class, 'tango') !== false
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param  OrderItem $orderItem
     * @return array
     */
    protected function getHawkOrderInfo(OrderItem $orderItem): array
    {
        $vendorOrderNumber = $orderItem->order->vendor_order_number;
        $clientProgramNumber = $orderItem->vendor_meta['payload']['clientProgramNumber'];
        $requestId = key($orderItem->vendor_meta['attempts']);
        return app(Client::class)
            ->orderInfo($vendorOrderNumber, $clientProgramNumber, $requestId)
            ->toArray();
    }

    /**
     * @param  OrderItem $orderItem
     * @return array
     */
    protected function getTangoOrderInfo(OrderItem $orderItem): array
    {
        return ['Info' => 'Coming soon...'];

        // $vendorOrderNumber = $orderItem->order->vendor_order_number;
        // $requestId = key($orderItem->vendor_meta['attempts']);
        // return app(Client::class)
        //     ->orderInfo($vendorOrderNumber, $requestId)
        //     ->toArray();
    }

    /**
     * @param  OrderItem $orderItem
     * @return boolean
     */
    protected function hasVendorMetaData(OrderItem $orderItem): bool
    {
        return isset($orderItem->vendor_meta['attempts']) && isset($orderItem->vendor_meta['payload']);
    }
}
