<?php

namespace Ignite\Catalog\Entities;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

/**
 * Catalog Entity
 *
 * @property int $id
 * @property string $name
 * @property string $code
 */
class Catalog extends Model
{
    const STATUS_ACTIVE = 1;
    const STATUS_INACTIVE = 0;

    /** @var string */
    protected $table = 'catalog';

    /** @var string */
    protected $primaryKey = 'id';

    /** @var array */
    protected $fillable = ['catalog_vendor_id', 'code', 'name', 'type', 'description', 'countries', 'active'];

    /**
     */
    public static function boot()
    {
        parent::boot();

        self::creating(function ($model) {
            static::generateCodeIfEmpty($model);
        });

        self::updating(function ($model) {
            static::generateCodeIfEmpty($model);
        });
    }

    /**
     * Set the vendor countries.
     *
     * @param array|string $countries
     */
    public function setCountriesAttribute($countries)
    {
        if (is_array($countries)) {
            $countries = implode(',', array_map('trim', $countries));
        }

        $this->attributes['countries'] = $countries;
    }

    /**
     * Set the vendor countries.
     *
     * @param mixed $flag
     */
    public function setActiveAttribute($flag)
    {
        $this->attributes['active'] = (bool) $flag;
    }

    /**
     * Is the catalog active.
     *
     * @return bool
     */
    public function isActive()
    {
        return (int) $this->active === 1;
    }

    /**
     * Is the catalog inactive.
     *
     * @return bool
     */
    public function isInactive()
    {
        return ! $this->isActive();
    }

    /**
     * The relationship to the vendor.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function vendor()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\Vendor::class, 'catalog_vendor_id', 'id');
    }

    /**
     * The relationship to the catalog items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\Item::class, 'catalog_id', 'id');
    }

    /**
     * The relationship to the catalog items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function menu()
    {
        return $this->hasOne(\Ignite\Catalog\Entities\Menu::class, 'catalog_id', 'id');
    }

    /**
     * Gets the URL to the start of the given catalog.
     *
     * @return string
     */
    public function url(): string
    {
        if (false !== stripos($this->code, 'helix')) {
            // @kludgy: needs to be in the Helix Catalog module or database instead of here
            $url = route('catalog.helix');
        } elseif (stripos($this->code, 'reloadable')) {
            $url = route('catalog.hawk-reloadable');
        } else {
            $url = route('catalog.show', ['catalog' => $this->code]);
        }

        return $url;
    }

    /**
     * Query for multiple IDs.
     *
     * @param  Builder      $query
     * @param  array|string $ids
     * @return Builder
     */
    public function scopeByIds(Builder $query, $ids)
    {
        if (is_string($ids)) {
            $ids = preg_split('/\s?,\s?/', $ids);
        }

        $query->whereIn('id', $ids);

        return $query;
    }

    /**
     * Only active catalog records.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     */
    public function scopeOnlyActive($query)
    {
        $query->where('active', static::STATUS_ACTIVE);
    }

    /**
     * Only inactive catalog records.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     */
    public function scopeOnlyInactive($query)
    {
        $query->where('active', static::STATUS_INACTIVE);
    }

    /**
     * Only catalogs supported by the current participant's country.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param \Ignite\Core\Entities\Participant $participant
     */
    public function scopeByParticipant($query, $participant)
    {
        $query->where('countries', 'like', "%$participant->country%");
    }

    /**
     * The data to be stored in the session.
     *
     * @return array
     */
    public function toSessionData()
    {
        return [
            'id' => $this->getKey(),
            'name' => $this->name,
            'code' => $this->code
        ];
    }

    /**
     * The presentable data for a catalog.
     *
     * @return array
     */
    public function toPresentableArray()
    {
        return [
            'status' => $this->isActive() ? 'Active' : 'Inactive',
            'type' => $this->type,
            'countries' => $this->countries,
            'description' => $this->description,
            'created_at' => $this->created_at->format('Y-m-d H:i:s'),
            'updated_at' => $this->updated_at->format('Y-m-d H:i:s'),
        ];
    }

    /**
     * Allow participant to access the catalog based on their country.
     *
     * @param  \Ignite\Core\Entities\User  $user
     * @return bool
     */
    public function allowParticipant($user)
    {
        $countries = explode(',', $this->countries);

        return in_array($user->participant->country, $countries);
    }

    /**
     * @param Catalog $model
     */
    protected static function generateCodeIfEmpty(Catalog $model): void
    {
        if (empty($model->code) && !empty($model->name)) {
            $model->code = str_slug($model->name, '-');
        }
    }
}
