<?php

namespace Ignite\Catalog\Providers;

use Ignite\Catalog\Models\Product\ProductTypeFactory;
use Ignite\Catalog\Scopes\ParticipantScope;
use Ignite\Core\Entities\Participant;
use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\Ssl;
use Ignite\Vendor\Hawk\UnsubscribeData;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Observers\OrderObserver;
use Ignite\Catalog\Repositories\CartRepository;
use Ignite\Catalog\Repositories\CategoryRepository;
use Ignite\Catalog\Repositories\MenuRepository;
use Ignite\Catalog\Repositories\AttributeRepository;
use Ignite\Catalog\Repositories\ItemRepository;
use Ignite\Catalog\Repositories\CatalogRepository;
use Ignite\Catalog\Repositories\FavoriteRepository;
use Ignite\Catalog\Repositories\OrderRepository;
use Ignite\Core\Repositories\TransactionRepository;
use Illuminate\Container\Container;
use Illuminate\Database\Eloquent\Factory;
use Illuminate\Foundation\Application;
use Illuminate\Support\ServiceProvider;

class CatalogServiceProvider extends ServiceProvider
{
    /**
     * The module name.
     * @var string
     */
    protected $moduleName = 'Catalog';

    /**
     * Indicates whether the views for the module can be published to the vendor directory.
     *
     * @var bool
     */
    protected $canPublishViews = false;

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->bootConfig();
        $this->bootViews();
        $this->bootTranslations();
        $this->bootFactories();
        $this->bootAssets();
        $this->loadMigrationsFrom(__DIR__ . '/../../database/migrations');

        Order::observe(OrderObserver::class);
    }

    /**
     * Publish assets for ignite.
     */
    protected function bootAssets()
    {
        $this->publishes([
            __DIR__.'/../../assets/ignite' => public_path('vendor/ignite'),
        ], 'ignite-catalog');
    }

    /**
     * Boot config.
     *
     * @return void
     */
    protected function bootConfig()
    {
        $configPath = __DIR__ . '/../../config/config.php';
        $configName = strtolower($this->moduleName);

        $this->publishes([
            "$configPath" => config_path($configName) . '.php'
        ], 'ignite-catalog');

        if (file_exists($configPath)) {
            $this->mergeConfigFrom($configPath, $configName);
        }
    }

    /**
     * Boot views.
     *
     * @return void
     */
    public function bootViews()
    {
        $sourcePath = __DIR__ . '/../../resources/views';

        $paths = [$sourcePath];

        if ($this->canPublishViews) {
            $paths = array_merge($paths, array_map(function ($path) {
                return $path . '/modules/' . $this->moduleName;
            }, $this->app['config']->get('view.paths')));
            $this->publishes([
                $sourcePath => base_path('resources/views/modules/' . $this->moduleName)
            ], 'ignite-catalog');
        }

        $this->loadViewsFrom($paths, $this->moduleName);
    }

    /**
     * Boot translations.
     *
     * @return void
     */
    public function bootTranslations()
    {
        $langPath = base_path('resources/lang/modules/' . strtolower($this->moduleName));

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleName);
        } else {
            $this->loadTranslationsFrom(__DIR__ . '/../../resources/lang', $this->moduleName);
        }
    }

    /**
     * Boot an additional directory of factories.
     *
     * @return void
     */
    public function bootFactories()
    {
        if (! app()->environment('production')) {
            app(Factory::class)->load(__DIR__ . '/../../database/factories');
        }
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        $this->app->register(RouteServiceProvider::class);
        $this->app->register(ViewServiceProvider::class);
        $this->app->register(EventServiceProvider::class);
        $this->app->register(ConsoleServiceProvider::class);

        $this->registerObservers();
        $this->registerRepositories();
        $this->registerRouteModelBindings();
        $this->registerHawkApi();
        $this->registerProductTypes();

        Participant::addGlobalScope(app(ParticipantScope::class));
    }

    /**
     * Register repository objects as singletons.
     */
    private function registerRepositories()
    {
        $this->app->singleton(\Ignite\Catalog\Contracts\VendorRepository::class, function () {
            return new \Ignite\Catalog\Repositories\VendorRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\CatalogRepository::class, function () {
            return new CatalogRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\CatalogItemRepository::class, function () {
            return new ItemRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\FavoriteRepository::class, function () {
            return new FavoriteRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\CartRepository::class, function () {
            return new CartRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\OrderRepository::class, function ($app) {
            return new OrderRepository(
                $app->make(TransactionRepository::class)
            );
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\AttributeRepository::class, function () {
            return new AttributeRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\CategoryRepository::class, function () {
            return new CategoryRepository();
        });

        $this->app->singleton(\Ignite\Catalog\Contracts\MenuRepository::class, function () {
            return new MenuRepository();
        });

        $this->app->singleton(Cart::class, function ($app) {
            $cart = $app->make(CartRepository::class);
            $transaction = $app->make(TransactionRepository::class);
            return new Cart(
                $cart->findByUser(),
                (float) $transaction->getBalance(),
                (float) $cart->total()
            );
        });
    }

    /**
     * Register model observers.
     */
    private function registerObservers()
    {
        Order::observe(OrderObserver::class);
    }

    /**
     * Register the product type factory.
     */
    public function registerProductTypes()
    {
        $this->app->singleton(ProductTypeFactory::class, function (Application $app) {
            $factory = new ProductTypeFactory($app);

            return $factory->registerTypes($app->config->get('catalog.product.types', []));
        });
    }

    /**
     * Register route model bindings
     */
    private function registerRouteModelBindings()
    {
        $this->app['router']->bind('catalog', function ($value) {
            return Catalog::where('code', $value)->first() ?? abort(404);
        });
    }

    /**
     * Register the Hawk API classes.
     */
    protected function registerHawkApi()
    {
        $this->app->singleton(Ssl::class, function (Container $app) {
            $isTest = $app['config']->get('catalog.vendors.hawk.test_mode');

            $cert = config('catalog.vendors.hawk.cert');
            if (!$cert || $cert === base_path()) {
                $cert = dirname(__FILE__) . '/../Support/certs/'
                    . ($isTest ? 'brightspot.integration.cert.pem' : 'brightspot.cert.pem');
            } elseif (!file_exists($cert)) {
                $cert = base_path($cert);
            }

            $key = config('catalog.vendors.hawk.key');
            if (!$key || $key === base_path()) {
                $key = dirname(__FILE__) . '/../Support/certs/'
                    . ($isTest ? 'brightspot.integration.key' : 'brightspot.key');
            } elseif (!file_exists($key)) {
                $key = base_path($key);
            }

            return new Ssl($cert, $key);
        });

        $this->app->singleton(UnsubscribeData::class, function (Container $app, array $params) {
            return new UnsubscribeData(
                isset($params['companyName']) ? $params['companyName'] : $app['config']->get("catalog.vendors.hawk.company"),
                isset($params['companyName']) ? $params['addressLine1'] : $app['config']->get("catalog.vendors.hawk.address1"),
                isset($params['addressLine1']) ? $params['addressCity'] : $app['config']->get("catalog.vendors.hawk.city"),
                isset($params['addressState']) ? $params['addressState'] :  $app['config']->get("catalog.vendors.hawk.state"),
                isset($params['addressZip']) ? $params['addressZip'] : $app['config']->get("catalog.vendors.hawk.zip"),
                isset($params['addressCountry']) ? $params['addressCountry'] : $app['config']->get("catalog.vendors.hawk.country"),
                isset($params['methodType']) ? $params['methodType'] : $app['config']->get("catalog.vendors.hawk.unsubscribe.type"),
                isset($params['value']) ? $params['value'] : $app['config']->get("catalog.vendors.hawk.unsubscribe.value")
            );
        });

        $this->app->singleton(EmailContent::class, function (Container $app, array $params) {
            return new EmailContent(
                isset($params['senderName']) ? $params['senderName'] : $app['config']->get("catalog.vendors.hawk.egift.senderName"),
                isset($params['senderEmail']) ? $params['senderEmail'] : $app['config']->get("catalog.vendors.hawk.egift.senderEmail"),
                isset($params['subject']) ? $params['subject'] : $app['config']->get("catalog.vendors.hawk.egift.subject"),
                isset($params['emailMessage']) ? $params['emailMessage'] : $app['config']->get("catalog.vendors.hawk.egift.emailMessage"),
                isset($params['emailClosingName']) ? $params['emailClosingName'] : $app['config']->get("catalog.vendors.hawk.egift.emailClosingName"),
                isset($params['unsubscribeData']) ? $params['unsubscribeData'] : app(UnsubscribeData::class)
            );
        });

        $this->app->singleton(Client::class, function (Container $app) {
            return new Client(
                $app['config']->get('catalog.vendors.hawk.merchant_id'),
                $app->make(Ssl::class),
                null,
                $app['config']->get('catalog.vendors.hawk.test_mode')
            );
        });
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
