<?php

namespace Ignite\Catalog\Console;

use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Entities\MenuItem;
use Ignite\Catalog\Entities\Option;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Core\Facades\Format;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

/**
 * Sample json for items with price:
 *   {
 *     "sku" : "amazon_egift_us",
 *     "catalog" : "gift-cards-us",
 *     "type": "simple",
 *     "class" : "egift",
 *     "locale" : "en",
 *     "name" : "Amazon.com eGift",
 *     "short_description" : null,
 *     "description" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "terms" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "manufacturer" : "Amazon.com",
 *     "image" : "amazon.png",
 *     "msrp" : 0,
 *     "cost" : 0,
 *     "price": 100,
 *     "price_markup": 0.15,
 *     "point_value" : 0.005,
 *     "points" : 23000,
 *     "visibility" : "1",
 *     "active" : "1",
 *     "vendor_active": 1,
 *     "vendor_meta" : {"content_provider_code": "Amazon_eGift"},
 *     "categories" : "General|Home & Garden"
 *   },
 *
 * Sample json for items with denominations:
 *   {
 *     "sku" : "amazon_egift_us",
 *     "catalog" : "gift-cards-us",
 *     "type": "configurable",
 *     "class" : "egift",
 *     "locale" : "en",
 *     "name" : "Amazon.com eGift",
 *     "short_description" : null,
 *     "description" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "terms" : "Lorem ipsum dolor sit amet, consectetuer adipiscing elit..",
 *     "manufacturer" : "Amazon.com",
 *     "image" : "amazon.png",
 *     "denominations" : "25,50,100,500",
 *     "price_markup": 0.15,
 *     "point_value" : 0.005,
 *     "visibility" : "1",
 *     "active" : "1",
 *     "vendor_active": 1,
 *     "vendor_meta" : {"content_provider_code": "Amazon_eGift"},
 *     "categories" : "General|Home & Garden"
 *   },
 *
 * Note: you can use unix commands like 'jq' or 'csvtojson'/'json2csv' to convert .csv files to json format and reverse.
 *
 * @todo: update sample file Catalog\src\Console\data\hawk\egift.json and Catalog\src\Console\data\hawk\giftcard.json
 */

class Hawk extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:hawk';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import Hawk catalog data into Ignite';

    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $dryRunPositiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * The collection of categories.
     *
     * @var \Illuminate\Support\Collection
     */
    protected $categories;

    /**
     * Catalogs, keyed by their codes.
     *
     * @var Catalog[]
     */
    protected $catalogs = [];

    /**
     * The catalog menus
     *
     * @var array
     */
    protected $menuIds;

    /**
     * The catalog vendor to attach the items.
     *
     * @var Vendor
     */
    protected $vendor;

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $product = $this->argument('product');
        $filename = $this->option('file');
        $denominations = $this->option('denominations');

        $data = $this->getJsonData($filename);

        if (empty($data)) {
            throw new \DomainException("No data available in $filename.");
        }

        // $this->clear();
        $this->process($data, $product, $denominations);

        return 0;
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->dryRunPositiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            ['product', InputArgument::REQUIRED, 'The product ID for this program.'],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['denominations', null, InputOption::VALUE_OPTIONAL, 'The denominations to create for each card.', null],
            ['point', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['markup', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['file', null, InputOption::VALUE_OPTIONAL, 'The file from which to import card data.', realpath(__DIR__ . "/data/hawk/giftcard.json")],
            // ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param $e
     */
    protected function handleException($e)
    {
        if ($this->output->isVerbose() || $this->output->isVeryVerbose() || $this->output->isDebug()) {
            throw $e;
        }

        $this->error(
            sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine())
        );
    }

    /**
     * The data from the JSON file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getJsonData($filename)
    {
        if (! starts_with($filename, '/')) {
            $filename = realpath(__DIR__ . "/data/hawk/$filename");
        }

        $json = file_get_contents($filename);

        $data = json_decode($json, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException("Unable to parse json in $filename. Error: " . json_last_error_msg());
        }

        if (! isset($data['data']) || ! is_array($data['data'])) {
            throw new \DomainException("Unable to find expected top-level key named 'data'.");
        }

        return $data['data'];
    }

    /**
     * Process the items.
     *
     * @param  array $data
     * @param  int   $product
     * @param  array $denominations
     * @return bool
     */
    protected function process($data, $product, $denominations)
    {
        $rows = collect($data);
        $categories = collect();

        $this->info(sprintf('Importing %s %s.', $count = $rows->count(), str_plural('card', $count)));

        // Create the 'denomination' attribute if it does not already exist.
        $denominationAttribute = $this->getOrCreateDenominationAttribute();

        // Extrapolate the source data into Catalog Item entities.
        foreach ($rows as $number => $row) {
            if (isset($row['categories']) && ! empty($row['categories'])) {
                $this->categories(
                    $categories = $this->updateOrCreateCategories($row['categories'])->keyBy('code')
                );
                $this->addCategoryMenuItems($row['catalog'], $categories);
                unset($row['categories']);
            }

            $row['code'] = str_slug($row['sku'], '-');

            // Create new Catalog Item instances for denominations.
            $simples = collect();
            $denominations = $row['denominations'] ?? $denominations;
            if (is_string($denominations)) {
                $denominations = explode(',', $denominations);
            }
            if (is_array($denominations)) {
                foreach ($denominations as $denomination) {
                    // Calculate points based on markup/margin.
                    $simples->push($this->createItemFromDenomination($denomination, $row, $product));
                }
            }

            $item = $this->createItem($row, $product);

            foreach ($simples as $index => $simple) {
                // Associate the simple items to the configurable as catalog_item_options.
                $item->options()->save(
                    $this->createOption($item, $simple, $denominationAttribute, $index)
                );
                // Associate the simple items to the configurable as catalog_attribute_items.
                $item->attributes()->associate(
                    $this->createAttributeItem($simple, $denominationAttribute, $index)
                );
            }

            // Associate both the simple and configurable items to the correct categories.
            $categories->each(function ($category) use ($item) {
                $item->categories()->detach($category);
            });
            foreach ($categories as $category) {
                $item->categories()->attach($category->getKey(), [
                    'position' => $number + 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            unset($item->item_id);

            $item->points_min = $simples->min('points');
            $item->points_max = $simples->max('points');

            $item->save();
        }

        $this->info('Done.');

        return true;
    }

    /**
     * Ensure we have a denomination attribute.
     *
     * @return Attribute
     */
    protected function getOrCreateDenominationAttribute()
    {
        return Attribute::updateOrCreate(['code' => 'denomination'], [
            'label' => 'Denomination',
            'code' => 'denomination',
            'type' => 'select'
        ]);
    }

    /**
     * Update or create categories.
     *
     * @param  string $categories
     * @return \Illuminate\Support\Collection
     */
    protected function updateOrCreateCategories($categories)
    {
        return collect(explode('|', $categories))->map(function ($category) {
            $code = $this->getCategoryCode($category);
            return Category::updateOrCreate(['code' => $code], [
                'code' => $code,
                'name' => $category,
                'active' => 1
            ]);
        });
    }

    /**
     * Get the category code from the category name.
     *
     * @param  string $category
     * @return string
     */
    protected function getCategoryCode($category)
    {
        return str_slug(str_replace(['&', '+'], 'and', $category));
    }

    /**
     * Get or merge the categories.
     *
     * @param  null|\Illuminate\Support\Collection $items
     * @return static
     */
    protected function categories($items = null)
    {
        if (is_null($this->categories)) {
            $this->categories = collect();
        }

        if (! is_null($items)) {
            foreach ($items as $item) {
                $this->categories->push($item);
            }
        }

        return $this->categories->keyBy('code');
    }

    /**
     * Create an item from a denomination value.
     *
     * @param  int $denomination
     * @param  array $row
     * @param  int $product
     * @return Item
     */
    protected function createItemFromDenomination($denomination, $row, $product)
    {
        $markup = (float) ($this->option('markup') ?? $row['price_markup']);
        $point = (float) ($this->option('point') ?? $row['point_value']);
        $denomination = (float) $denomination;

        $priceMarkedUp = $denomination + ($denomination * $row['price_markup']);

        $row['msrp'] = $denomination;
        $row['cost'] = $denomination;
        $row['price'] = $denomination;
        $row['price_markup'] = $markup;
        $row['price_margin'] = ($priceMarkedUp - $denomination) / $priceMarkedUp;
        $row['point_value'] = $point;
        $row['points'] = $this->calculatePoints($denomination, $markup, $point);
        $row['sku'] = $row['sku'] . '_' . $denomination;
        $row['code'] = $row['code'] . '-' . $denomination;
        $row['type'] = "simple";
        $row['name'] = sprintf('%s $%s', $row['name'], $denomination);
        $row['description'] = '';
        $row['terms'] = '';
        $row['visibility'] = 0;

        return $this->createItem($row, $product);
    }

    /**
     * Create an item.
     *
     * @param  array  $row
     * @param  string $product
     * @return Item
     */
    protected function createItem($row, $product)
    {
        if (! isset($row['catalog_vendor_id'])) {
            $row['catalog_vendor_id'] = $this->vendor()->getKey();
        }

        if (! isset($row['catalog_id'])) {
            $row['catalog_id'] = $this->catalog($row['catalog'])->getKey();
        }

        if (! isset($row['msrp'])) {
            $row['msrp'] = 0.0000;
        }

        if (! isset($row['price'])) {
            $row['price'] = 0.0000;
        }

        if (! isset($row['points'])) {
            $row['points'] = 0.0000;
        }

        if (! isset($row['vendor_meta'])) {
            $row['vendor_meta'] = [];
        }

        $row['vendor_meta']['product_id'] = $product;

        if (isset($row['product_id'])) {
            unset($row['product_id']);
        }
        if (isset($row['denominations'])) {
            unset($row['denominations']);
        }
        if (isset($row['catalog'])) {
            unset($row['catalog']);
        }

        echo "{$row['name']} ({$row['sku']})\n";

        return Item::updateOrCreate(['sku' => $row['sku']], $row);
    }

    /**
     * Calculate the points from the price, markup and point value.
     *
     * @param  float $denomination
     * @param  float $markup
     * @param  float $point
     * @return float
     */
    protected function calculatePoints($denomination, $markup, $point)
    {
        return round($denomination * (1 + $markup) / $point, 0, PHP_ROUND_HALF_UP);
    }

    /**
     * Create the option.
     *
     * @param  Item      $item
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     *
     * @return Option
     */
    protected function createOption($item, $simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'super_id' => $item->getKey(),
            'item_id' => $simple->getKey(),
            'attribute_id' => $attribute->getKey(),
            'value' => $value,
        ];

        $data = array_merge($where, [
            'label' => '$' . $value,
            'position' => $index + 1
        ]);

        return Option::updateOrCreate($where, $data);
    }

    /**
     * Create the association to the attribute.
     *
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     * @return AttributeItem
     */
    protected function createAttributeItem($simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'attribute_id' => $attribute->getKey(),
            'item_id' => $simple->getKey(),
            'value' => $value
        ];

        $data = [
            'position' => $index + 1,
            'active' => 1
        ];

        return AttributeItem::updateOrCreate($where, $data);
    }

    /**
     * @param string     $catalog
     * @param Collection $categories
     */
    protected function addCategoryMenuItems(string $catalog, Collection $categories)
    {
        $catalogMenuId = $this->menuId($catalog);
        foreach ($categories as $category) {
            MenuItem::updateOrCreate([
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
            ], [
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
                "active" => 1,
                "parent_id" => 0,
                "position" => $category->id,
            ]);
        }
    }

    /**
     * @param  string $catalog
     * @return int
     */
    protected function menuId(string $catalog)
    {
        if (is_null($this->menuIds)) {
            $this->menuIds = Menu::all()->pluck('id', 'catalog_id')->toArray();
        }

        $catalogId = $this->catalog($catalog)->getKey();

        return $this->menuIds[$catalogId];
    }

    /**
     * The catalog entity.
     *
     * @return Catalog
     * @throws \Exception
     */
    protected function catalog($code)
    {
        if (! isset($this->catalogs[$code])) {
            $this->catalogs[$code] = Catalog::where('code', $code)->first();
        }

        if ($this->catalogs[$code]) {
            return $this->catalogs[$code];
        }

        throw new \Exception('No catalogs exist. Run `module:seed` first.');
    }

    /**
     * The catalog vendor entity.
     *
     * @return Vendor
     * @throws \Exception
     */
    protected function vendor()
    {
        if (is_null($this->vendor)) {
            $this->vendor = Vendor::where('name', 'hawk')->first();
            if (is_null($this->vendor)) {
                throw new \Exception('No vendors exist. Run `module:seed` first.');
            }
        }

        return $this->vendor;
    }

    /**
     */
    protected function clear()
    {
        \DB::statement('SET FOREIGN_KEY_CHECKS = 0');
        \DB::delete('TRUNCATE TABLE `catalog_category_item`');
        \DB::delete('TRUNCATE TABLE `catalog_menu_item`');
        \DB::delete('TRUNCATE TABLE `catalog_category`');
        \DB::delete('TRUNCATE TABLE `catalog_item_option`');
        \DB::delete('TRUNCATE TABLE `catalog_item`');
        \DB::statement('SET FOREIGN_KEY_CHECKS = 1');
        \DB::statement('ALTER TABLE `catalog_category_item` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_menu_item` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_category` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item_option` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item` AUTO_INCREMENT = 100');
    }
}
