<?php

namespace Ignite\Catalog\Console;

use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Entities\MenuItem;
use Ignite\Catalog\Entities\Option;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Core\Facades\Format;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class CatalogItemImportCommandTrait extends Command
{
    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $dryRunPositiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * The collection of categories.
     *
     * @var \Illuminate\Support\Collection
     */
    protected $categories;

    /**
     * Catalogs, keyed by their codes.
     *
     * @var Catalog[]
     */
    protected $catalogs = [];

    /**
     * The catalog menus
     *
     * @var array
     */
    protected $menuIds;

    /**
     * The catalog vendor to attach the items.
     *
     * @var Vendor
     */
    protected $vendor;

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $filename = $this->option('file');
        $isReset = $this->option('reset');
        $data = $this->getCsvData($filename);

        if (empty($data)) {
            throw new \DomainException("No data available in $filename.");
        }

        // if $isReset and not on production
        if ($isReset && app()->environment() === 'production') {
            $this->error('Resetting the catalog database on production is not allowed with this.');
            return 1;
        }
        if ($isReset) {
            $this->clear();
        }

        $this->process($data);

        return 0;
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->dryRunPositiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            // ['product', InputArgument::REQUIRED, 'The product ID for this program.'],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['file', 'f', InputOption::VALUE_OPTIONAL, 'The file from which to import card data.', 'giftcards.csv'],
            ['reset', 'r', InputOption::VALUE_NONE, 'Truncates catalog related tables; not available on production.'],
            // ['dry-run', null, InputOption::VALUE_NONE, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param $e
     */
    protected function handleException($e)
    {
        if ($this->output->isVerbose() || $this->output->isVeryVerbose() || $this->output->isDebug()) {
            throw $e;
        }

        $this->error(
            sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine())
        );
    }

    /**
     * The data from the JSON file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getJsonData(string $filename)
    {
        if (! starts_with($filename, '/')) {
            $filename = storage_path(__DIR__ . "/data/hawk/$filename");
        }

        $json = file_get_contents($filename);
        $data = json_decode($json, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException("Unable to parse json in $filename. Error: " . json_last_error_msg());
        }

        if (! isset($data['data']) || ! is_array($data['data'])) {
            throw new \DomainException("Unable to find expected top-level key named 'data'.");
        }

        return $data['data'];
    }

    /**
     * The data from the CSV file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getCsvData(string $filename)
    {
        if (! starts_with($filename, '/')) {
            $filename = storage_path($filename);
        }

        $fileHandle = fopen($filename, 'r');
        $data = [];
        $headers = fgetcsv($fileHandle, 4096, ',');
        // remove BOM if exists
        $headers = array_map(function ($value) {
            return preg_replace('/^[\x{FEFF}]+/u', '', trim($value));
        }, $headers);

        while (($row = fgetcsv($fileHandle, 4096, ',')) !== FALSE) {
            // trim each value in row
            $row = array_map('trim', $row);
            $data[] = array_combine($headers, $row);
        }
        fclose($fileHandle);

        return $data;
    }

    /**
     * Process the items.
     *
     * @param  array $data
     * @return bool
     */
    protected function process(array $data)
    {
        $rows = collect($data);
        $categories = collect();

        $this->introMessage($rows);

        // Create the 'denomination' attribute if it does not already exist.
        $denominationAttribute = $this->getOrCreateDenominationAttribute();

        // Extrapolate the source data into Catalog Item entities.
        foreach ($rows as $number => $row) {
            $row = $this->prepareRow($number, $row);

            if (isset($row['categories']) && ! empty($row['categories'])) {
                // @todo: do we want to create the category,
                // or throw an error if it is an unknown category?
                $this->categories(
                    $categories = $this->updateOrCreateCategories($row['categories'])->keyBy('code')
                );
                $this->addCategoryMenuItems($row['catalog'], $categories);
                unset($row['categories']);
            }

            $row['code'] = str_slug($row['sku'], '-');

            // Create new Catalog Item instances for denominations.
            $simples = collect();
            $denominations = $row['denominations'] ?? null;
            if (is_string($denominations)) {
                $denominations = explode(',', $denominations);
            }
            if (is_array($denominations)) {
                foreach ($denominations as $denomination) {
                    // Calculate points based on markup/margin.
                    $simples->push($this->createItemFromDenomination($denomination, $row));
                }
            } else {
                $simples->push($this->createItemFromDenomination($row['price'], $row));
            }

            $item = $this->createItem($row);

            foreach ($simples as $index => $simple) {
                // Associate the simple items to the configurable as catalog_item_options.
                $item->options()->save(
                    $this->createOption($item, $simple, $denominationAttribute, $index)
                );
                // Associate the simple items to the configurable as catalog_attribute_items.
                $item->attributes()->associate(
                    $this->createAttributeItem($simple, $denominationAttribute, $index)
                );
            }

            // Associate both the simple and configurable items to the correct categories.
            $categories->each(function ($category) use ($item) {
                $item->categories()->detach($category);
            });
            foreach ($categories as $category) {
                $item->categories()->attach($category->getKey(), [
                    'position' => $number + 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            unset($item->item_id);

            $item->points_min = $simples->min('points');
            $item->points_max = $simples->max('points');

            $item->save();
        }

        $this->info('Done.');

        return true;
    }

    /**
     * @param  [type] $rows
     * @return void
     */
    protected function introMessage($rows)
    {
        $this->info(sprintf('Importing %s %s.', $count = $rows->count(), str_plural('card', $count)));
        $this->warn('Points and price for each item will be re-calculated based on the denomination or cost, exchange rate, and markup.');
    }

    /**
     * Prepare the row
     *
     * @param  int    $number
     * @param  array  $row
     * @return array
     */
    protected function prepareRow(int $number, array $row): array
    {
        return $row;
    }

    /**
     * Ensure we have a denomination attribute.
     *
     * @return Attribute
     */
    protected function getOrCreateDenominationAttribute()
    {
        return Attribute::updateOrCreate(['code' => 'denomination'], [
            'label' => 'Denomination',
            'code' => 'denomination',
            'type' => 'select'
        ]);
    }

    /**
     * Update or create categories.
     *
     * @param  string $categories
     * @return \Illuminate\Support\Collection
     */
    protected function updateOrCreateCategories(string $categories): Collection
    {
        return collect(explode('|', $categories))->map(function ($category) {
            $code = $this->getCategoryCode($category);
            return Category::updateOrCreate(['code' => $code], [
                'code' => $code,
                'name' => $category,
                'active' => 1
            ]);
        });
    }

    /**
     * Get the category code from the category name.
     *
     * @param  string $category
     * @return string
     */
    protected function getCategoryCode(string $category): string
    {
        return str_slug(str_replace(['&', '+'], 'and', $category));
    }

    /**
     * Get or merge the categories.
     *
     * @param  null|\Illuminate\Support\Collection $items
     * @return static
     */
    protected function categories($items = null)
    {
        if (is_null($this->categories)) {
            $this->categories = collect();
        }

        if (! is_null($items)) {
            foreach ($items as $item) {
                $this->categories->push($item);
            }
        }

        return $this->categories->keyBy('code');
    }

    /**
     * Split $denomination into amount::name if we can
     *
     * @param  string $denomination
     * @return array
     */
    protected function splitDenomination(string $denominationString): array
    {
        if (preg_match('/^([0-9.]+)::(.+)$/', $denominationString, $matches)) {
            $denomination = (float) $matches[1];
            $name = $matches[2];
        } else {
            $denomination = (float) $denominationString;
            $name = $denominationString;
        }

        return [$name, $denomination];
    }

    /**
     * Create an item from a denomination value.
     *
     * @param  string $denomination
     * @param  array  $row
     * @return Item
     */
    protected function createItemFromDenomination(string $denominationString, array $row)
    {
        list($name, $denomination) = $this->splitDenomination($denominationString);

        $markup = (float) $row['price_markup'] ?? 0;
        $pointValue = (float) config('catalog.default_point_value');
        $usdAmount = $this->exchangeToUsd($denomination, $row['exchange_to_usd']);

        $row['msrp'] = empty($row['msrp']) ? $usdAmount : $row['msrp'];
        $row['cost'] = empty($row['cost']) ? $usdAmount : $row['cost'];
        $row['price'] = empty($row['price']) ? $this->calculatePrice($row['cost'], $markup) : $row['price'];
        $row['price_markup'] = $markup;
        $row['price_margin'] = ($row['price'] - $row['cost']) / $row['price'];
        $row['point_value'] = $pointValue;
        $row['points'] = $this->calculatePoints($usdAmount, $markup, $pointValue);
        $row['sku'] = $row['sku'] . '_' . str_slug($name, '_');
        $row['code'] = $row['code'] . '-' . str_slug($name, '-');
        $row['type'] = 'simple';
        $row['name'] = sprintf(
            '%s %s%s',
            $row['name'],
            is_numeric($name) && 1 == $row['exchange_to_usd'] ? '$' : '- ',
            $name
        );
        $row['short_description'] = '';
        $row['description'] = '';
        $row['terms'] = '';
        $row['disclaimer'] = '';
        $row['visibility'] = 0;

        $row['vendor_meta']['amount'] = $denomination;
        if ($name != $denomination) {
            $row['vendor_meta']['amount_label'] = $name;
        }

        return $this->createItem($row);
    }

    /**
     * Create an item.
     *
     * @param  array  $row
     * @return Item
     */
    protected function createItem(array $row)
    {
        if (! isset($row['catalog_vendor_id'])) {
            $row['catalog_vendor_id'] = $this->vendor()->getKey();
        }

        if (! isset($row['catalog_id'])) {
            $row['catalog_id'] = $this->catalog($row['catalog'])->getKey();
        }

        if (! isset($row['msrp']) || empty($row['msrp'])) {
            $row['msrp'] = 0.00;
        }

        if (! isset($row['cost']) || empty($row['cost'])) {
            $row['cost'] = 0.00;
        }

        if (! isset($row['price']) || empty($row['price'])) {
            $row['price'] = 0.00;
        }

        if (! isset($row['points']) || empty($row['points'])) {
            // should this be an error?
            $row['points'] = 0.00;
        }

        if (! isset($row['vendor_meta']) || empty($row['vendor_meta'])) {
            $row['vendor_meta'] = [];
        }

        // truncate short_description if more than 250 chars
        $row['short_description'] = strip_tags($row['short_description']);
        if (isset($row['short_description']) && strlen($row['short_description']) > 250) {
            $row['short_description'] = substr(
                $row['short_description'],
                0,
                strrpos(substr($row['short_description'], 0, 250), ' ')
            ) . '...';
        }

        if (! isset($row['short_description'])) {
            $row['vendor_meta'] = [];
        }

        $allowed = [
            'catalog_vendor_id', 'catalog_id', 'code', 'sku', 'type', 'class', 'name',
            'locale', 'short_description', 'description', 'terms', 'disclaimer',
            'manufacturer', 'image', 'msrp', 'cost', 'price', 'exchange_to_usd',
            'price_markup', 'price_margin', 'point_value', 'points', 'points_min',
            'points_max', 'visibility', 'active', 'vendor_active', 'vendor_meta',
        ];
        $row = array_intersect_key($row, array_flip($allowed));

        echo "{$row['name']} ({$row['sku']})\n";

        return Item::updateOrCreate(['sku' => $row['sku']], $row);
    }

    /**
     * Convert the amount to USD based on some exchange rate.
     *
     * @param  float $denomination
     * @param  float $exchangeToUsd
     * @return float
     */
    protected function exchangeToUsd(float $denomination, float $exchangeToUsd)
    {
        return $denomination / $exchangeToUsd;
    }

    /**
     * Calculate the price from the cost and markup.
     *
     * @param  float $usdCost
     * @param  float $markup
     * @return float
     */
    protected function calculatePrice($usdCost, $markup)
    {
        return $usdCost * (1 + $markup);
    }

    /**
     * Calculate the points from the price, markup and point value.
     *
     * @param  float $usdCost
     * @param  float $markup
     * @param  float $usDollarPerPoint
     * @return float
     */
    protected function calculatePoints($usdCost, $markup, $usDollarPerPoint)
    {
        return ceil($usdCost * (1 + $markup) / $usDollarPerPoint);
    }

    /**
     * Create the option.
     *
     * @param  Item      $item
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     *
     * @return Option
     */
    protected function createOption($item, $simple, $attribute, $index)
    {
        $amount = $simple->vendor_meta['amount'] > 0 ? $simple->vendor_meta['amount'] : $simple->price;
        $value = isset($simple->vendor_meta['amount_label'])
            ? $simple->vendor_meta['amount_label']
            : (1 == $simple->exchange_to_usd ? '$' : '') . Format::amount($amount);

        $where = [
            'super_id' => $item->getKey(),
            'item_id' => $simple->getKey(),
            'attribute_id' => $attribute->getKey(),
            'value' => $value,
        ];

        $data = array_merge($where, [
            'label' => $value,
            'position' => $index + 1
        ]);

        return Option::updateOrCreate($where, $data);
    }

    /**
     * Create the association to the attribute.
     *
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     * @return AttributeItem
     */
    protected function createAttributeItem($simple, $attribute, $index)
    {
        $amount = $simple->vendor_meta['amount'] > 0 ? $simple->vendor_meta['amount'] : $simple->price;
        $value = isset($simple->vendor_meta['amount_label'])
            ? $simple->vendor_meta['amount_label']
            : (1 == $simple->exchange_to_usd ? '$' : '') . Format::amount($amount);

        $where = [
            'attribute_id' => $attribute->getKey(),
            'item_id' => $simple->getKey(),
            'value' => $value,
        ];

        $data = [
            'position' => $index + 1,
            'active' => 1
        ];

        return AttributeItem::updateOrCreate($where, $data);
    }

    /**
     * @param string     $catalog
     * @param Collection $categories
     */
    protected function addCategoryMenuItems(string $catalog, Collection $categories)
    {
        $catalogMenuId = $this->menuId($catalog);
        foreach ($categories as $category) {
            MenuItem::updateOrCreate([
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
            ], [
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
                "active" => 1,
                "parent_id" => 0,
                "position" => $category->id,
            ]);
        }
    }

    /**
     * @param  string $catalog
     * @return int
     */
    protected function menuId(string $catalog)
    {
        if (is_null($this->menuIds)) {
            $this->menuIds = Menu::all()->pluck('id', 'catalog_id')->toArray();
        }

        $catalogId = $this->catalog($catalog)->getKey();

        return $this->menuIds[$catalogId];
    }

    /**
     * The catalog entity.
     *
     * @return Catalog
     * @throws \Exception
     */
    protected function catalog($code)
    {
        if (! isset($this->catalogs[$code])) {
            $this->catalogs[$code] = Catalog::where('code', $code)->first();
        }

        if ($this->catalogs[$code]) {
            return $this->catalogs[$code];
        }

        throw new \Exception('No catalogs exist. Run `module:seed` first.');
    }

    /**
     * The catalog vendor entity.
     *
     * @return Vendor
     * @throws \Exception
     */
    protected function vendor()
    {
        if (is_null($this->vendor)) {
            $this->vendor = Vendor::where('name', 'hawk')->first();
            if (is_null($this->vendor)) {
                throw new \Exception('No vendors exist. Run `module:seed` first.');
            }
        }

        return $this->vendor;
    }

    /**
     */
    protected function clear()
    {
        $this->alert('Clearing the catalog item tables!');

        \DB::statement('SET FOREIGN_KEY_CHECKS = 0');
        \DB::delete('TRUNCATE TABLE catalog_attribute');
        \DB::delete('TRUNCATE TABLE catalog_attribute_item');
        \DB::delete('TRUNCATE TABLE catalog_category');
        \DB::delete('TRUNCATE TABLE catalog_category_item');
        \DB::delete('TRUNCATE TABLE catalog_item');
        \DB::delete('TRUNCATE TABLE catalog_item_option');
        // \DB::delete('TRUNCATE TABLE catalog_menu');
        \DB::delete('TRUNCATE TABLE catalog_menu_item');
        \DB::delete('TRUNCATE TABLE catalog_favorite');
        \DB::delete('TRUNCATE TABLE catalog_cart_item');
        \DB::statement('SET FOREIGN_KEY_CHECKS = 1');

        \DB::statement('ALTER TABLE `catalog_category_item` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_menu_item` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_category` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item_option` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item` AUTO_INCREMENT = 100');
    }
}
