<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Catalog\Emails\OrderItemFailure;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Events\CompleteHawkOrder;
use Ignite\Vendor\Hawk\Response\BaseOrderResponse;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\Response;
use Illuminate\Support\Facades\Mail;

class ProcessManager
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * @var string
     */
    protected $requestId;

    /**
     * ProcessManager constructor.
     *
     * @param Order $order
     * @param OrderItem $orderItem
     * @param $requestId
     */
    public function __construct(Order $order, OrderItem $orderItem, $requestId)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
        $this->requestId = $requestId;
    }

    /**
     * Complete the order.
     *
     * @param Response $response
     * @param array $meta
     *
     * @return bool
     */
    public function complete(Response $response, array $meta = [])
    {
        $this->process($response, [
            'processed' => 1,
            'processed_at' => $now = now(),
            'processed_quantity' => $qty = $this->order->totalQuantity(),
            'shipped' => 1,
            'shipped_at' => $now,
            'shipped_quantity' => $qty,
            'cancelled' => 0,
            'cancelled_at' => null,
            'cancelled_quantity' => 0,
        ], $meta);

        $this->order->refresh();

        if ($this->order->hasProcessedAllItems()) {
            $this->order->update([
                'processed' => 1,
                'processed_at' => $now,
                'cancelled' => 0,
                'cancelled_at' => null,
            ]);
        }

        CompleteHawkOrder::dispatch($this->order, $this->orderItem, $response, $this->requestId);

        return true;
    }

    /**
     * Cancel the order.
     *
     * @param Response $response
     * @param array $meta
     *
     * @return bool
     */
    public function cancel(Response $response, array $meta = [])
    {
        $this->process($response, [
            'processed' => 0,
            'processed_at' => null,
            'processed_quantity' => 0,
            'shipped' => 0,
            'shipped_at' => null,
            'shipped_quantity' => 0,
            'cancelled' => 1,
            'cancelled_at' => $now = now(),
            'cancelled_quantity' => $this->order->totalQuantity(),
        ], $meta);

        if ($this->order->hasCancelledAllItems()) {
            $this->order->update([
                'processed' => 0,
                'processed_at' => null,
                'cancelled' => 1,
                'cancelled_at' => $now,
            ]);
        }

        return true;
    }

    /**
     * Store the attempt.
     *
     * @param Response $response
     * @param array $meta
     *
     * @return bool
     */
    public function attempt(Response $response, array $meta = [])
    {
        return $this->process($response, [], $meta);
    }

    /**
     * Report the error.
     *
     * @param ErrorResponse $response
     * @param array $meta
     *
     * @return bool
     */
    public function error(ErrorResponse $response, array $meta = [])
    {
        // We're not cancelling the item yet, instead, we'll send an email with the reason.
        $this->process($response, [], $meta);

        Mail::send(new OrderItemFailure($this->order, $this->orderItem));

        return false;
    }

    /**
     * Process the response.
     *
     * @param Response $response
     * @param array $data
     * @param array $meta
     *
     * @return bool
     */
    protected function process(Response $response, array $data = [], array $meta = [])
    {
        $existing = $this->orderItem->vendor_meta ?? [];

        if (! isset($existing['attempts'])) {
            $existing['attempts'] = [];
        }

        $existing['attempts'][$this->requestId] = [
            'response' => $response->toArray(),
            'item' => $this->orderItem->item->vendor_meta ?? []
        ];

        if (array_key_exists('attempts', $meta)) {
            unset($meta['attempts']);
        }

        $this->orderItem->update(array_merge([
            'vendor_meta' => array_merge($existing, $meta),
        ], $data));

        return true;
    }

    /**
     * Determine whether the order response is complete.
     *
     * @param BaseOrderResponse $response
     *
     * @return bool
     */
    public function isOrderComplete(BaseOrderResponse $response)
    {
        return $response->getIsCompleted() && $response->getSuccess();
    }
}
