<?php

namespace Ignite\Catalog\Models\Report;

use Ignite\Catalog\Models\Grid\OrderItemFormatter;
use Ignite\Core\Entities\User;
use Ignite\Core\Models\Grid\QueryTable;
use Illuminate\Database\DatabaseManager;
use Yajra\DataTables\DataTables;

class OrderItemReport extends QueryTable
{
    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'username', 'company'
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'user_id'
    ];

    /**
     * @var OrderItemFormatter
     */
    protected $formatter;

    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        OrderItemFormatter $formatter,
        array $params = [])
    {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Order Date',
                'name' => 'catalog_order.created_at',
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'email' => [
                'title' => 'E-mail',
                'name' => 'participant.email',
                'orderable' => true,
                'exportable' => true,
            ],
            'number' => [
                'title' => 'Order #',
                'name' => 'catalog_order.number',
                'orderable' => true,
                'exportable' => true,
            ],
            'name' => [
                'title' => 'Item Name',
                'name' => 'catalog_order_item.name',
                'orderable' => true,
                'exportable' => true,
            ],
            'class' => [
                'title' => 'Item Type',
                'name' => 'catalog_order_item.class',
                'orderable' => true,
                'exportable' => true,
            ],
            'points' => [
                'title' => 'Points',
                'name' => 'catalog_order_item.points',
                'orderable' => true,
                'exportable' => true,
            ],
            'price' => [
                'title' => 'Price',
                'name' => 'catalog_order_item.price',
                'orderable' => true,
                'exportable' => true,
            ],
            'quantity' => [
                'title' => 'Quantity',
                'name' => 'catalog_order_item.quantity',
                'orderable' => true,
                'exportable' => true,
            ],
            'total' => [
                'title' => 'Total',
                'name' => 'total',
                'orderable' => true,
                'exportable' => true,
                'searchable' => false,
            ],
            'processed' => [
                'title' => 'Processed',
                'name' => 'catalog_order_item.processed',
                'orderable' => true,
                'exportable' => true,
            ],
            'ssn' => [
                'title' => 'SSN',
                'name' => 'participant.ssn',
                'visible' => false,
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())
            ->keyBy('name')
            ->except('total')
            ->keys()
            ->toArray();

        return $this->getConnection()
            ->table('catalog_order_item')
            ->select($columns)
            ->addSelect($this->getConnection()->raw('(catalog_order_item.points * catalog_order_item.quantity) as total'))
            ->leftJoin('catalog_order', 'catalog_order.id', '=', 'catalog_order_item.catalog_order_id')
            ->leftJoin('core_participant as participant', 'participant.user_id', '=', 'catalog_order.user_id')
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->orderBy('catalog_order.created_at', 'desc')
            ->where('user.internal', User::TYPE_PARTICIPANT);
    }

    /**
     * @inheritdoc
     */
    protected function getColumnFormattingMap()
    {
        return [
            'points' => [$this->formatter, 'points'],
            'total' => [$this->formatter, 'total'],
            'processed' => [$this->formatter, 'processed']
        ];
    }
}
