<?php

namespace Ignite\Catalog\Presenters;

use Ignite\Core\Facades\Format;
use Ignite\Packages\Presenter\Presenter;
use Ignite\Core\Presenters\CommonlyPresentable;
use Ignite\Core\Traits\Markdownable;

/**
 * ItemPresenter
 *
 * @property \Ignite\Catalog\Entities\Item $entity
 */
class ItemPresenter extends Presenter
{
    use Markdownable, CommonlyPresentable;

    /**
     * The name in a readable format.
     *
     * @return string
     */
    public function getName()
    {
        return trim($this->entity->name);
    }

    /**
     * Class name in a redable format
     *
     * @return string
     */
    public function getClass(): string
    {
        $class = explode('.', $this->entity->class);
        if (is_array($class)) {
            return reset($class);
        }

        return $class;
    }

    /**
     * The sku in a readable format.
     *
     * @return string
     */
    public function getSku()
    {
        return trim($this->entity->sku);
    }

    /**
     * The sku in a readable format.
     *
     * @return string
     */
    public function getType()
    {
        return trim($this->entity->type);
    }

    /**
     * The classification.
     *
     * @return string
     */
    public function getClassification()
    {
        return trim($this->entity->class);
    }

    /**
     * Show the badge on the item.
     *
     * @return bool
     */
    public function showBadge()
    {
        return $this->entity->isCardType() && $this->getClassification();
    }

    /**
     * The canonical item url.
     *
     * @param  array $append
     * @return string
     */
    public function getUrl($append = [])
    {
        return route('catalog.item.show', array_merge($append, ['item' => $this->entity->getKey()]));
    }

    /**
     * The url to favorite this item.
     *
     * @return string
     */
    public function getFavoriteUrl()
    {
        if (is_null($this->entity->favorite)) {
            return route('catalog.favorites.store', ['item' => $this->entity->getKey()]);
        }

        return route('catalog.favorites.destroy', ['item' => $this->entity->getKey()]);
    }

    /**
     * The description excerpt.
     *
     * @param  int $limit
     * @return string
     */
    public function getExcerpt($limit = 25)
    {
        if (! empty($short = $this->entity->short_description)) {
            return $short;
        }

        $description = $this->entity->description;
        return str_replace("\n", "<br />\n", substr($description, 0, $limit)) . ((strlen($description) > $limit) ? '...' : '');
    }

    /**
     * Format the description.
     *
     * @param  bool $plain
     * @return string
     */
    public function getDescription($plain = false)
    {
        $description = $this->entity->description;

        return ($plain) ? $description : $this->markdown($description);
    }

    /**
     * Format the terms.
     *
     * @param  bool $plain
     * @return string
     */
    public function getTerms($plain = false)
    {
        $terms = $this->entity->terms;

        return ($plain) ? $terms : $this->markdown($terms);
    }

    /**
     * Get the active status in a readable format.
     *
     * @return string
     */
    public function getActiveStatus()
    {
        return $this->formatYesNo($this->entity->active);
    }

    /**
     * Get the vendor name in a readable format.
     *
     * @return string
     */
    public function getVendorName()
    {
        return ucwords(snake_case($this->entity->vendor->name, ' '));
    }

    /**
     * Get the active status in a readable format.
     *
     * @return string
     */
    public function getVendorStatus()
    {
        return $this->formatYesNo($this->entity->vendor_active);
    }

    /**
     * Get the visibility in a readable format.
     *
     * @return string
     */
    public function getVisibility()
    {
        return (bool) $this->entity->visibility ? trans('Core::core.common.visible') : trans('Core::core.common.invisible');
    }

    /**
     * Get the msrp in a readable format.
     *
     * @return string
     */
    public function getMsrp()
    {
        return $this->formatCurrency(Format::amount($this->entity->msrp));
    }

    /**
     * Get the cost in a readable format.
     *
     * @return string
     */
    public function getCost()
    {
        return $this->formatCurrency(Format::amount($this->entity->cost));
    }

    /**
     * Get the point value in a readable format.
     *
     * @return string
     */
    public function getPointValue()
    {
        return number_format($this->entity->point_value, 2);
    }

    /**
     * Get the price in a readable format.
     *
     * @return string
     */
    public function getPrice()
    {
        return $this->formatCurrency(Format::amount($this->entity->price));
    }

    /**
     * Get points in a readable format.
     *
     * @param bool $withLabel
     * @return string
     */
    public function getPoints($withLabel = true)
    {
        $label = trans_choice('Core::core.common.points', 2);

        if ($this->entity->isConfigurable()) {
            $points = $this->entity->getTypeInstance()->getPointsRange();
        } else {
            $points = $this->entity->getPoints();
        }

        // Should we be running Format::amount() on $points here? Looking at these, seems like we shouldn't be:
        // \Ignite\Catalog\Models\Product\Types\Configurable::getPointsRange()
        // \Ignite\Catalog\Entities\Item::getPoints()
        return new \Illuminate\Support\HtmlString(
            Format::amount($points).($withLabel ? " $label" : '')
        );
    }

    /**
     * Get the price margin in a readable format.
     *
     * @return string
     */
    public function getPriceMargin()
    {
        return $this->formatPercentage($this->entity->price_margin, 2);
    }

    /**
     * Get the price margin in a readable format.
     *
     * @return string
     */
    public function getPriceMarkup()
    {
        return $this->formatPercentage($this->entity->price_markup, 2);
    }
}
