<?php

namespace Ignite\Catalog\Http\Controllers;

use Exception;
use Ignite\Flash\Facades\Flash;
use Illuminate\Http\RedirectResponse;
use Ignite\Catalog\Contracts\FavoriteRepository;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Http\Controllers\Controller;
use Illuminate\View\View;

class FavoritesController extends Controller
{
    /**
     * @var FavoriteRepository
     */
    private $favoriteRepository;

    /**
     * FavoritesController constructor.
     *
     * @param FavoriteRepository $favoriteRepository
     */
    public function __construct(FavoriteRepository $favoriteRepository)
    {
        $this->favoriteRepository = $favoriteRepository;
    }

    /**
     * List all the users' favorite items.
     *
     * @return View
     */
    public function index()
    {
        $items = $this->favoriteRepository->paginate(24);

        return view('catalog.favorites.index', compact('items'));
    }

    /**
     * Add the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return RedirectResponse
     */
    public function store(Item $item)
    {
        try {
            $this->favoriteRepository->favorite($item);
            $message = __('Catalog::theme.favorite.flash.add.success');
            $status = 'success';
        } catch (Exception $exception) {
            logger()->critical('Unable to add item to favorites', compact('exception'));
            $message = __('Catalog::theme.favorite.flash.add.exception');
            $status = 'error';
        }

        Flash::add($message, $status);

        return redirect()->back();
    }

    /**
     * Destroy the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return RedirectResponse
     */
    public function destroy(Item $item)
    {
        try {
            $this->favoriteRepository->unfavorite($item);
            $message = __('Catalog::theme.favorite.flash.remove.success');
            $status = 'success';
        } catch (Exception $exception) {
            logger()->critical('Unable to remove item from favorites.', compact('exception'));
            $message = __('Catalog::theme.favorite.flash.remove.exception');
            $status = 'error';
        }

        Flash::add($message, $status);

        return redirect()->back();
    }
}
