<?php

namespace Ignite\Catalog\Models\Grid;

use Ignite\Core\Facades\Format;
use Illuminate\Support\HtmlString;
use Ignite\Core\Models\Grid\Formatter;

class OrderFormatter extends Formatter
{
    /**
     * Format a checkbox for selecting entries in the grid.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function checkbox($model)
    {
        return $this->formatCheckbox($model->getKey());
    }

    /**
     * Custom column for actions.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function actions($model)
    {
        $actions = [
            [
                'click' => '',
                'class' => '',
                'icon' => 'eye-open',
                'label' => 'View',
                'text' => 'text-default',
                'type' => 'default',
                'link' => $model->present()->getAdminShowUrl()
            ],
        ];

        return $this->buttons($actions);
    }

    /**
     * Format the items column.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function items($model)
    {
        return $this->html($model->present()->getAssociatedItemNames());
    }

    /**
     * Format an order number with a link.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function number($model)
    {
        return $this->html(sprintf(
            '<a href="%s">%s</a>',
            $model->present()->getAdminShowUrl(),
            $model->present()->getNumber()
        ));
    }

    /**
     * Format an email with a link.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function email($model)
    {
        return $this->html(sprintf(
            '<a href="mailto:%1$s">%1$s</a>',
            $model->present()->getShippingEmail()
        ));
    }

    /**
     * Format a participant on an order.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function participant($model)
    {
        return $this->html(sprintf(
            '<a href="%s"><span class="fa fa-user"></span> %s</a>',
            $model->present()->getAdminParticipantShowUrl(),
            $model->present()->getParticipantName()
        ));
    }

    /**
     * Format whether an order has been processed.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function processed($model)
    {
        return $this->html($model->present()->getProcessedState());
    }

    /**
     * Format whether an order has been cancelled.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function cancelled($model)
    {
        return $this->html($model->present()->getCancelledState());
    }

    /**
     * Format whether an order has been processed.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     * @throws \Ignite\Packages\Presenter\Exceptions\PresenterException
     */
    public function status($model)
    {
        if ($model->isProcessing()) {
            $icon = $model->present()->getProcessingIcon();
        } elseif ($model->isProcessed()) {
            $icon = $model->present()->getProcessedIcon();
        } elseif ($model->isCancelled()) {
            $icon = $model->present()->getCancelledIcon();
        } else {
            $icon = trans('Core::core.common.unknown');
        }

        return $this->html($icon);
    }

    /**
     * Format the points for an order.
     *
     * @param  \Ignite\Catalog\Entities\Order $model
     * @return HtmlString
     */
    public function points($model)
    {
        return $this->html(Format::amount($model->points));
    }
}
