(function($, window, document, undefined) {

  'use strict';

  function Calculator(options) {
    this.options = $.extend({
      refs: {
        cost: '#cost',
        price: '#price',
        margin: '#price_margin',
        markup: '#price_markup',
        value: '#point_value',
        points: '#points',
      }
    }, options);
  }

  Calculator.prototype = {
    recalculate: function(props, changed) {
      props = this._parse(this._fill(props, changed));
      for (var prop in props) {
        if (props.hasOwnProperty(prop) && prop !== changed) {
          props = this[prop].apply(this, [props]);
        }
      }
      this.commit(props);
    },
    commit: function(props) {
      for (var prop in props) {
        if (props.hasOwnProperty(prop)) {
          this._commit(prop, props[prop]);
        }
      }
    },
    cost: function(props) {
      this._commit('cost', props.cost, false);
      return props;
    },
    price: function(props) {
      props.price = this._calculatePrice(props);
      return props;
    },
    margin: function(props) {
      props.margin = this._calculateMargin(props);
      return props;
    },
    markup: function(props) {
      props.markup = this._calculateMarkup(props);
      return props;
    },
    points: function(props) {
      props.points = this._calculatePoints(props);
      return props;
    },
    value: function(props) {
      props.points = this._calculatePoints(props);
      return props;
    },
    _commit: function(name, value) {
      var $field = $(this.options.refs[name]);
        $field.val(value);

        if (name === 'points') {
            $field.val(value.toFixed(0));
        }

        if(name === 'price') {
            $field.val(value.toFixed(2));
        }

      $field.trigger('change');
    },
    _fill: function(props, changed) {
      for (var key in this.options.refs) {
        if (key !== changed) {
          props[key] = $(this.options.refs[key]).val();
        }
      }
      return props;
    },
    _parse: function(props) {
      for (var key in props) {
        props[key] = parseFloat(props[key]);
      }
      return props;
    },
    _calculateMargin: function(props) {
      var markup = props.markup / 100;
      var margin = markup / (1 + markup);
      return margin * 100;
    },
    _calculateMarkup: function(props) {
      var margin = props.margin / 100;
      var markup = margin / (1 - margin);
      return markup * 100;
    },
    _calculatePrice: function(props) {
      if (props.value <= 0) {
        return props.cost;
      }
      return props.cost + ((props.cost * props.markup) / 100);
    },
    _calculatePoints: function(props) {
      if (parseFloat(props.value) <= 0) {
        return 0;
      }
      return Math.round((props.price * 100) / (props.value * 100));
    }
  };

  var pluginName = "calculator",
    defaults = {
      field: '',
      fieldSuffix: '-text-value',
      fieldClass: 'rangeslider-label',
      calculator: {}
    };

  // The actual plugin constructor
  function Plugin (element, options) {
    this.element = element;
    this.settings = $.extend({}, defaults, options);
    this._defaults = defaults;
    this._name = pluginName;
    this.init();
  }

  // Avoid Plugin.prototype conflicts
  $.extend( Plugin.prototype, {
    init: function() {
      var calculator = new Calculator(this.settings.calculator);
      var updatableRefs = [
        calculator.options.refs.cost,
        calculator.options.refs.price,
        calculator.options.refs.value,
        calculator.options.refs.points,
      ];

      $('#contrain-pricing').on('change', function (evt) {
        if (evt.target.checked === false) {
          $(calculator.options.refs.value).removeAttr('readonly');
          $(calculator.options.refs.points).removeAttr('readonly');
        } else {
          $(calculator.options.refs.value).attr('readonly', 'readonly');
          $(calculator.options.refs.points).attr('readonly', 'readonly');
        }
      });

      $(updatableRefs.join(',')).on('keyup', function(evt) {
        if (!$('#contrain-pricing').prop('checked')) {
            return
        };
        var props = {};
        var prop = evt.target.id === 'point_value' ? 'value' : evt.target.id;
        props[prop] = $(this).val();
        calculator.recalculate(props, prop);
      });

      var name = this.settings.field;
      var $field = $(this.element);
      var $fieldText = $('<div></div>', {
        id: name + this.settings.fieldSuffix,
        class: this.settings.fieldClass
      });

      var checkValue = function(value) {
        // Check if value matches program default
        var $check = $('.slider-value-check');
        if ($check.length) {
          var defaultValue = parseFloat($check.attr('data-default'));
          if (value === defaultValue) {
            $check.hide();
          } else {
            $check.show();
          }
        }
      };

      $fieldText.insertBefore($field);
      $field.rangeslider({
        polyfill: false,
        onInit: function() {
          $fieldText.text(parseFloat($field.val()).toFixed(2) + '%');
        },
        onSlide: function(position, value) {
          $fieldText.text(parseFloat(value).toFixed(2) + '%');
          checkValue(value);
        },
        onSlideEnd: function(position, value) {
          var props = {};
          props[name] = value;
          $fieldText.text(parseFloat(value).toFixed(2) + '%');
          calculator.recalculate(props, name);
          checkValue(value);
        }
      });
    }
  });

  $.fn[pluginName] = function(options) {
    return this.each(function() {
      if (! $.data( this, "plugin_" + pluginName)) {
        $.data(this, "plugin_" + pluginName, new Plugin(this, options));
      }
    });
  };

})(jQuery, window, document);
