<?php

namespace Ignite\Catalog\Presenters;

use Ignite\Core\Facades\Format;
use Ignite\Packages\Presenter\Presenter;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Core\Presenters\CommonlyPresentable;
use Ignite\Core\Traits\Markdownable;

/**
 * OrderPresenter
 *
 * @property \Ignite\Catalog\Entities\Order $entity
 */
class OrderPresenter extends Presenter
{
    use Markdownable, CommonlyPresentable;

    /**
     * The admin url to show the order.
     *
     * @return string
     */
    public function getAdminShowUrl()
    {
        return route('admin.catalog.orders.show', ['number' => $this->entity->number]);
    }

    /**
     * The admin url to process the order.
     *
     * @return string
     */
    public function getAdminProcessUrl()
    {
        return route('admin.catalog.orders.process', ['number' => $this->entity->number]);
    }

    /**
     * The admin url to cancel the order.
     *
     * @return string
     */
    public function getAdminCancelUrl()
    {
        return route('admin.catalog.orders.cancel', ['number' => $this->entity->number]);
    }

    /**
     * The admin url to show the related participant.
     *
     * @return string
     */
    public function getAdminParticipantShowUrl()
    {
        return route('admin.participant.show', $this->entity->user_id);
    }

    /**
     * The id of the related participant.
     *
     * @return int
     */
    public function getParticipantId()
    {
        return $this->entity->user_id;
    }

    /**
     * The name of the related participant.
     *
     * @return string
     */
    public function getParticipantName()
    {
        return ucwords($this->entity->participant->fullName());
    }

    /**
     * The order number.
     *
     * @return string
     */
    public function getNumber()
    {
        return "#{$this->entity->number}";
    }

    /**
     * Calculate the total points for the order.
     *
     * @return string
     */
    public function totalPoints()
    {
        $total = $this->entity->items->reduce(function ($previous, OrderItem $orderItem) {
            return $previous + ($orderItem->points * $orderItem->quantity);
        }, 0);


        return Format::amount($total);
    }

    /**
     * Calculate the total quantity of items in the order.
     *
     * @return string
     */
    public function totalQuantity()
    {
        return Format::amount($this->entity->items->sum('quantity'));
    }

    /**
     * The name of the person to whom we should ship.
     *
     * @return string
     */
    public function getShippingName()
    {
        return $this->entity->ship_name ?? '';
    }

    /**
     * The email of the person to whom we should ship.
     *
     * @return string
     */
    public function getShippingEmail()
    {
        return $this->entity->ship_email ?? '';
    }

    /**
     * The shipping address line 1.
     *
     * @return string
     */
    public function getShippingAddress1()
    {
        return $this->entity->ship_address_1 ?? '';
    }

    /**
     * The shipping address line 2.
     *
     * @return string
     */
    public function getShippingAddress2()
    {
        return $this->entity->ship_address_2 ?? '';
    }

    /**
     * The shipping address line 3.
     *
     * @return string
     */
    public function getShippingAddress3()
    {
        return $this->entity->ship_address_3 ?? '';
    }

    /**
     * The shipping city.
     *
     * @return string
     */
    public function getShippingCity()
    {
        return $this->entity->ship_city ?? '';
    }

    /**
     * The shipping state.
     *
     * @return string
     */
    public function getShippingState()
    {
        return $this->entity->ship_state ?? '';
    }

    /**
     * The shipping postal.
     *
     * @return string
     */
    public function getShippingPostal()
    {
        return $this->entity->ship_postal ?? '';
    }

    /**
     * The shipping country.
     *
     * @return string
     */
    public function getShippingCountry()
    {
        return $this->entity->ship_country ?? 'US';
    }

    /**
     * The shipping phone.
     *
     * @return string
     */
    public function getShippingPhone()
    {
        return $this->entity->ship_phone ?? '';
    }

    /**
     * The names of the items associated to this order.
     *
     * @return string
     */
    public function getAssociatedItemNames()
    {
        return $this->entity->items->map(function ($item) {
            return $item->name;
        })->implode(', ');
    }

    /**
     * Is the order processed.
     *
     * @return string
     */
    public function getProcessedState()
    {
        return $this->formatYesNo($this->entity->processed);
    }

    /**
     * Is the order cancelled.
     *
     * @return string
     */
    public function getCancelledState()
    {
        return $this->formatYesNo($this->entity->cancelled);
    }

    /**
     * Format a processing icon.
     *
     * @return string
     */
    public function getProcessingIcon()
    {
        return sprintf(
            '<span class="label label-warning">%1$s</span>',
            trans('Catalog::order.misc.status.processing.label')
        );
    }

    /**
     * Format a processed icon.
     *
     * @return string
     */
    public function getProcessedIcon()
    {
        return sprintf(
            '<span class="label label-success">%1$s</span>',
            trans('Catalog::order.misc.status.processed.label')
        );
    }

    /**
     * Format a cancelled icon.
     *
     * @return string
     */
    public function getCancelledIcon()
    {
        return sprintf(
            '<span class="label label-danger">%1$s</span>',
            trans('Catalog::order.misc.status.cancelled.label')
        );
    }
}
