<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\OpenLoopAnonymousOrderLine;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopAnonymousRequest;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopAnonymousResponse;
use Ignite\Vendor\Hawk\ShippingMethod;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class PrepaidAnonymousBulk extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitOpenLoopAnonymous(
            $payload, $requestId
        );
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        return $this->buildOpenLoopAnonymous();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return SubmitOpenLoopAnonymousResponse::class;
    }

    /**
     * The OpenLoopPersonalizedIndividual data.
     *
     * @return SubmitOpenLoopAnonymousRequest
     */
    protected function buildOpenLoopAnonymous()
    {
        $data = [
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines()
        ];

        return SubmitOpenLoopAnonymousRequest::fromAssociativeArray($data);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines(): array
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int)$this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return OpenLoopAnonymousOrderLine
     */
    protected function buildOrderLine(): OpenLoopAnonymousOrderLine
    {
        $data = [
            'description' => config('catalog.vendors.hawk.prepaid.anonymous.bulk.description'),
            'clientRefId' => config('catalog.vendors.hawk.prepaid.anonymous.bulk.clientRefId'),
            'fourthLineEmbossText' => 'Congratulations!',
            'cardCarrierMessage' => config('catalog.vendors.hawk.prepaid.card_carrier_message', ''),
            'shippingMethod' => $this->getShippingMethod(),
            'amount' => (int)$this->orderItem->cost,
            'deliveryAddress' => $this->getRecipientData(),
            'quantity' => (int)$this->orderItem->quantity
        ];

        return OpenLoopAnonymousOrderLine::fromAssociativeArray($data);
    }

    /**
     * @return string
     */
    protected function getShippingMethod(): string
    {
        $className = $this->getClassKey();

        return config("catalog.vendors.hawk.{$className}.shipping_method") ?? ShippingMethod::UPS_GROUND;
    }

    /**
     * The recipient data from the order.
     *
     * @return array
     */
    protected function getRecipientData(): array
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return [
            'company' => $this->order->user->participant->company,
            'firstName' => $first,
            'lastName' => $last,
            'line1' => $this->order->ship_address_1,
            'line2' => $this->order->ship_address_2 ?? null,
            'line3' => $this->order->ship_address_3 ?? null,
            'city' => $this->order->ship_city,
            'region' => $this->order->ship_state,
            'postalCode' => $this->order->ship_postal,
            'country' => $this->getCountry(),
            'postalCodeExt' => ''
        ];
    }
}
