<?php

namespace Ignite\Catalog\Tests\Unit\Models\Hawk;

use Ignite\Catalog\Models\Hawk\ClosedLoopBulkOrderLine;
use Ignite\Catalog\Models\Hawk\ClosedLoopIndividualOrderLine;
use Ignite\Catalog\Models\Hawk\EgiftIndividualOrderLine;
use Ignite\Catalog\Models\Hawk\EmailContent;
use Ignite\Catalog\Models\Hawk\PaymentType;
use Ignite\Catalog\Models\Hawk\Request\SubmitClosedLoopBulkRequest;
use Ignite\Catalog\Models\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Catalog\Models\Hawk\Request\SubmitEgiftIndividualRequest;
use Ignite\Catalog\Models\Hawk\Response\SubmitClosedLoopBulkResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Catalog\Models\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Catalog\Models\Hawk\ShippingMethod;
use Ignite\Catalog\Models\Hawk\UnsubscribeData;
use Ignite\Catalog\Models\Hawk\VirtualRecipient;
use Ignite\Catalog\Services\HawkApi;
use Ignite\Catalog\Tests\TestCase;

class HawkApiTestFoo extends TestCase
{
    const UNIVERSAL_VISA = '43476908';
    const CLOSED_LOOP_US_PHYSICAL_BULK = '43507608';
    const CLOSED_LOOP_US_PHYSICAL_INDIVIDUAL = '43507508';
    const CLOSED_LOOP_US_EGIFT_INDIVIDUAL = '43479508';

    /** @var HawkApi */
    protected $api;

    /**
     * Get the test configured instance of the HawkApi.
     *
     * @return HawkApi
     */
    private function getHawkApi()
    {
        $path = realpath(__DIR__ . "/../../../../../storage/certs");
        config()->set('catalog.vendors.hawk.merchant_id', "SNA9PY92ZJWWRW36X07G3YTRGW");
        config()->set('catalog.vendors.hawk.cert', "$path/brightspot.integration.cert.pem");
        config()->set('catalog.vendors.hawk.key', "$path/brightspot.integration.key");

        return app(HawkApi::class);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group OpenLoop
     * @group ApiCommunication
     * @throws \Ignite\Catalog\Vendors\ResponseException
     */
    public function it_can_send_an_egift_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_EGIFT_INDIVIDUAL;

        $request = SubmitEGiftIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                EgiftIndividualOrderLine::fromAssociativeArray([
                    'recipient' => VirtualRecipient::fromAssociativeArray([
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'email' => 'johnsmith@gmail.com',
                        //'id' => '12345',
                    ]),
                    'amount' => 10,
                    'contentProvider' => "AmericanEagleOutfitters_eGift",
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
            'returnCardNumberAndPIN' => true,
            'emailContent' => EmailContent::fromAssociativeArray([
                'senderName' => 'Ignite Platform',
                'senderEmail' => 'it@gobrightspot.com',
                'subject' => 'Test eGift Subject',
                'emailMessage' => 'Test eGift Message',
                'emailClosingName' => 'Test eGift Closing Name',
                'unsubscribeData' => UnsubscribeData::fromAssociativeArray([
                    'companyName' => 'Brightspot',
                    'addressLine1' => '102 Decker Court',
                    'addressCity' => 'Irving',
                    'addressState' => 'TX',
                    'addressZip' => '75062',
                    'addressCountry' => 'US',
                    'methodType' => 'EMAIL',
                    'value' => 'it@brightspot.email',
                ])
            ]),
        ]);

        $response = $this->getHawkApi()->submitEgiftIndividual($request, $requestId);

        $this->assertInstanceOf(SubmitEGiftIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     * @group ApiCommunication
     * @throws \Ignite\Catalog\Vendors\ResponseException
     */
    public function it_can_send_a_closed_loop_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_PHYSICAL_INDIVIDUAL;

        $request = SubmitClosedLoopIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                ClosedLoopIndividualOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'address' => [
                            'line1' => '102 Decker Ct',
                            'city' => 'Irving',
                            'region' => 'TX',
                            'postalCode' => '75062',
                            'country' => "USA",
                            'line2' => 'Suite 150'
                        ],
                    ],
                    'amount' => 100,
                    'quantity' => 5,
                    'contentProvider' => "Amazoncom",
                    'description' => 'The description text',
                    'groupingId' => '1',
                    'shippingMethod' => ShippingMethod::USPS_STANDARD_MAIL,
                    'cardCarrierMessage' => null,
                    'fourthLineEmbossText' => null,
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::ACH_DEBIT,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getHawkApi()->submitClosedLoopIndividual($request, $requestId);

        $this->assertInstanceOf(SubmitClosedLoopIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     * @group ApiCommunication
     * @throws \Ignite\Catalog\Vendors\ResponseException
     */
    public function it_can_send_a_closed_loop_bulk_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_PHYSICAL_BULK;

        $request = SubmitClosedLoopBulkRequest::fromAssociativeArray([
            'orderDetails' => [
                ClosedLoopBulkOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'address' => [
                            'line1' => '102 Decker Ct',
                            'city' => 'Irving',
                            'region' => 'TX',
                            'postalCode' => '75062',
                            'country' => "USA",
                            'line2' => 'Suite 150'
                        ],
                    ],
                    'deliveryAddress' => [
                        'company' => 'Brightspot',
                        'line1' => '102 Decker Ct',
                        'city' => 'Irving',
                        'region' => 'TX',
                        'postalCode' => '75062',
                        'country' => 'USA',
                        'line2' => 'Suite 150',
                        'line3' => '',
                    ],
                    'amount' => 100,
                    'quantity' => 5,
                    'contentProvider' => "Amazoncom",
                    'description' => 'The description text',
                    'groupingId' => '1',
                    'shippingMethod' => ShippingMethod::USPS_STANDARD_MAIL,
                    'cardCarrierMessage' => null,
                    'fourthLineEmbossText' => null,
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getHawkApi()->submitClosedLoopBulk($request, $requestId);

        $this->assertInstanceOf(SubmitClosedLoopBulkResponse::class, $response);
    }
}
