<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Contracts\FavoriteRepository as FavoriteRepositoryContract;
use Ignite\Catalog\Entities\Favorite;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Entities\User;

class FavoriteRepository implements FavoriteRepositoryContract
{
    /**
     * The most basic query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Favorite::query();
    }

    /**
     * The query scoped for a user.
     *
     * @param  User|null $user
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function queryByUser(User $user = null)
    {
        $query = $this->query();

        if ($user instanceof User) {
            $query = $query->byUser($user);
        } else {
            $query = $query->byCurrentUser();
        }

        return $query;
    }

    public function queryItemsByUser(User $user = null)
    {
        $query = $this->queryByUser($user);

        $items = $query->select(['item_id'])->orderBy('created_at', 'desc')->get()->keyBy('item_id')->toArray();

        $ids = array_keys($items);

        return Item::query()
            ->leftJoin('catalog_favorite', function (JoinClause $join) {
                $join->on('catalog_favorite.item_id', '=', 'catalog_item.id');
                $join->on('catalog_favorite.user_id', '=', DB::raw(auth()->user()->getKey()));
            })
            ->addSelect('catalog_item.*', 'catalog_favorite.id as favorite')
            ->onlyVisible()
            ->whereIn('catalog_item.id', $ids)
            ->orderByRaw('FIELD(catalog_item.id,' . implode(',', $ids) .')');
    }

    /**
     * Paginate the favorited items for the given user.
     *
     * @param  int $perPage
     * @param  User|null $user
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function paginate($perPage = 25, User $user = null)
    {
        return $this->queryItemsByUser($user)->paginate($perPage);
    }

    /**
     * Find all favorited items for a single user.
     *
     * @param  User|null $user
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function findByUser(User $user = null)
    {
        return $this->queryItemsByUser($user)->get();
    }

    /**
     * Find a favorite by id.
     *
     * @param  int $id
     * @param  string $column
     * @param  User|null $user
     * @return mixed
     */
    public function find($id, $column = 'id', User $user = null)
    {
        return $this->queryByUser($user)->where($column, $id)->first();
    }

    /**
     * Mark an item as a favorite.
     *
     * @param Item $item
     * @param User|null $user
     *
     * @return \Ignite\Catalog\Entities\Favorite
     */
    public function favorite(Item $item, User $user = null)
    {
        $data = [
            'item_id' => $item->getKey(),
            'user_id' => auth()->user()->getKey(),
        ];

        return Favorite::updateOrCreate($data, $data);
    }

    /**
     * Un-mark an item as a favorite.
     *
     * @param Item $item
     * @param User|null $user
     *
     * @return \Ignite\Catalog\Entities\Favorite
     */
    public function unfavorite(Item $item, User $user = null)
    {
        $favorite = $this->find($item->getKey(), 'item_id', $user);

        optional($favorite)->delete();

        return $favorite;
    }

    /**
     * Count the number of favorited items for a user.
     *
     * @param  User|null $user
     *
     * @return int
     */
    public function count(User $user = null)
    {
        return $this->queryByUser($user)->count();
    }
}
