<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Database\Connection;
use Illuminate\Support\Collection;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\CategoryItem;
use Ignite\Catalog\Entities\MenuItem;

class CategoryRepository implements \Ignite\Catalog\Contracts\CategoryRepository
{
    /**
     * The default query builder.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Category::query();
    }

    /**
     * Find a category by id.
     *
     * @param  int  $id
     * @return Category
     */
    public function find($id)
    {
        return Category::findOrFail($id);
    }

    /**
     * Create a category.
     *
     * @param  Collection  $data
     * @return Category
     */
    public function create(Collection $data)
    {
        $category = new Category($data->except('image')->toArray());

        if ($image = request()->file('image')) {
            $category->attachImage($image);
        }

        $category->save();

        return $category;
    }

    /**
     * Update a category.
     *
     * @param  int  $id
     * @param  Collection  $data
     * @return Category
     */
    public function update(int $id, Collection $data)
    {
        $category = $this->find($id);

        if ($image = request()->file('image')) {
            $category->attachImage($image);
        }

        $category->update($data->except('image')->toArray());

        return $category;
    }

    /**
     * Delete a category and any associated data.
     *
     * @param  int  $id
     * @return Category
     */
    public function delete(int $id)
    {
        $category = $this->find($id);
        $category->menuItems()->delete();
        $category->categoryItems()->delete();
        $category->forceDelete();

        return $category;
    }

    /**
     * Delete many categories and any associated data.
     *
     * @param  array  $ids
     * @return array
     */
    public function deleteMany(array $ids)
    {
        /** @var Connection $db */
        return app('db')->transaction(function () use ($ids) {
            MenuItem::query()->whereIn('catalog_category_id', $ids)->delete();
            CategoryItem::query()->whereIn('category_id', $ids)->delete();
            return Category::query()->whereIn('id', $ids)->delete();
        });
    }
}
