<?php

namespace Ignite\Catalog\Console;

use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Mail;
use Ignite\Catalog\Contracts\OrderRepository;
use Ignite\Catalog\Emails\OrderChecked;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class OrderCheck extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:order-check';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check all orders in a state of processing and based on their items status, determine if they can be marked as cancelled or processed.';

    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Flag to determine if we should send an email.
     *
     * @var bool
     */
    protected $shouldSendEmail;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $positiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * Execute the console command.
     *
     * @param OrderRepository $orderRepository
     * @return mixed
     */
    public function handle(OrderRepository $orderRepository)
    {
        $orders = $this->getProcessingOrders();

        $this->updateProcessedOrders($orders, $orderRepository);
        $this->updateCancelledOrders($orders, $orderRepository);
        $this->notifyProgramAdminOfRemainingProcessingOrders($orders);

        return 0;
    }

    /**
     * Update the processed orders.
     *
     * @param Collection $orders
     * @param OrderRepository $orderRepository
     */
    protected function updateProcessedOrders(Collection $orders, OrderRepository $orderRepository)
    {
        $this->info('Finding orders in processing state where all items are marked as processed...');

        $processed = $orders->filter(function (Order $order) {
            return $order->items->count() === $order->items->filter(function (OrderItem $item) {
                return $item->isProcessed();
            })->count();
        });

        $this->info('Found ' . $processed->count() . ' orders that can be updated to processed.');

        $processed->each(function (Order $order) use ($orderRepository) {
            if (! $this->isDryRun()) {
                $orderRepository->process($order->number);
            }
        });

        $this->info('Updated ' . $processed->count() . ' orders to processed.');
    }

    /**
     * Update the cancelled orders.
     *
     * @param Collection $orders
     * @param OrderRepository $orderRepository
     */
    protected function updateCancelledOrders(Collection $orders, OrderRepository $orderRepository)
    {
        $this->info('Finding orders in processing state where all items are marked as cancelled...');

        $cancelled = $orders->filter(function (Order $order) {
            return $order->items->count() === $order->items->filter(function (OrderItem $item) {
                return $item->isCancelled();
            })->count();
        });

        $this->info('Found ' . $cancelled->count() . ' orders that can be updated to cancelled.');

        $cancelled->each(function (Order $order) use ($orderRepository) {
            if (! $this->isDryRun()) {
                $orderRepository->cancel($order->number);
            }
        });

        $this->info('Updated ' . $cancelled->count() . ' orders to cancelled.');
    }

    /**
     * Notify program admin about orders that remain in a state of processing for
     * manual intervention.
     *
     * @param \Illuminate\Database\Eloquent\Collection $orders
     */
    protected function notifyProgramAdminOfRemainingProcessingOrders($orders)
    {
        $processing = $orders->filter(function (Order $order) {
            return $order->items->filter(function (OrderItem $item) {
                return $item->isProcessing();
            })->count() > 0;
        });

        $this->info('Found ' . $processing->count() . ' orders with items still marked as processing.');

        if ($this->shouldSendEmail()) {
            $this->info('Sending an email to ' . config('mail.from.address') . ' with all processing orders.');

            $processing = $processing->map(function (Order $order) {
                return $order->items->filter(function (OrderItem $item) {
                    return $item->isProcessing();
                });
            });

            Mail::send(new OrderChecked($processing->flatten()));
        }
    }

    /**
     * The orders in a state of processing.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getProcessingOrders()
    {
        return Order::processing()->get();
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->positiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * Determine if user asked to send an email.
     *
     * @return bool
     */
    protected function shouldSendEmail()
    {
        if (is_null($this->shouldSendEmail)) {
            $this->shouldSendEmail = in_array($this->option('send-email'), $this->positiveValues, true);
        }
        return $this->shouldSendEmail;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['send-email', null, InputOption::VALUE_OPTIONAL, 'Should we send an email?', null],
            ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param  \Exception $e
     * @throws \Exception
     */
    protected function handleException(\Exception $e)
    {
        if ($this->output->isVerbose() || $this->output->isVeryVerbose() || $this->output->isDebug()) {
            throw $e;
        }

        $this->error(
            sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine())
        );
    }
}
