<?php

namespace Ignite\Catalog\Models\Dashboard\Charts;

use Ignite\Core\Facades\Format;
use Ignite\Core\Models\Address;
use Illuminate\Support\Facades\DB;
use Ignite\Core\Models\Dashboard\Table;
use Illuminate\Support\Fluent;
use Illuminate\Support\HtmlString;

class MostPopularStatesTable extends Table
{
    /**
     * Create a new stat instance.
     *
     * @param  array|object $attributes
     */
    public function __construct($attributes = [])
    {
        $attributes = array_merge([
            'label'   => 'Most orders by state',
            'color'   => 'primary',
            'headers' => $this->getHeaders(),
            'keys'    => $this->getKeys(),
        ], $attributes);

        parent::__construct($attributes);
    }

    /**
     * The HTML element ID.
     *
     * @return string
     */
    public function getElement()
    {
        return 'popular-states-list';
    }

    /**
     * The chart size in the layout.
     *
     * @return string
     */
    public function getSize()
    {
        return $this->get('size', 'col-xs-12 col-sm-4');
    }

    /**
     * Get the correct data value from storage.
     *
     * @return mixed
     */
    public function getData()
    {
        if (! $this->get('data', false)) {
            $this->attributes['data'] = $this->buildRangedValue($this->getDateRange())->get()
                ->map(function ($row) {
                    return new Fluent([
                        'state' => $this->formatState($row),
                        'count' => $this->formatCount($row),
                    ]);
                });
        }

        return $this->get('data');
    }

    protected function buildRangedValue($range)
    {
        DB::statement("SET SESSION sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))");

        return DB::table('catalog_order')
            ->select([
                'ship_state',
                DB::raw('count(id) as `amount`')
            ])->selectSub(
                DB::table('catalog_order')->selectRaw('count(id)')->whereBetween('created_at', $range),
                'total'
            )
            ->whereBetween('created_at', $range)
            ->groupBy('ship_state')
            ->orderBy('amount', 'desc')
            ->take(10);
    }

    protected function getOrderItems($start, $end)
    {
        DB::statement("SET SESSION sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))");

        return DB::query()
          ->select(
              DB::raw('COUNT(catalog_favorite.item_id) as total'),
              'catalog_favorite.item_id as item_id',
              DB::raw('CONCAT(catalog_vendor.image_path, "/", catalog_item.image) as image'),
              'catalog_item.name as name'
          )
          ->from('catalog_order')
          ->leftJoin('catalog_item', 'catalog_favorite.item_id', '=', 'catalog_item.id')
          ->leftJoin('catalog_vendor', 'catalog_item.catalog_vendor_id', '=', 'catalog_vendor.id')
          ->whereBetween('catalog_favorite.created_at', [$start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s')])
          ->groupBy('catalog_favorite.item_id', 'catalog_item.name', 'catalog_vendor.image_path')
          ->orderBy('total', 'desc')
          ->take(10)
          ->get();
    }

    /**
     * The table headers, if any.
     *
     * @return array
     */
    public function getHeaders()
    {
        return [];
    }

    public function getKeys()
    {
        return ['count', 'state'];
    }

    protected function calculatePercentage($amount, $total)
    {
        $total = $total ?? 1;
        $amount = $amount ?? 1;
        $total = ($total < 1) ? 1 : $total;

        return Format::amount(($amount / $total) * 100, 2);
    }

    protected function formatState($row)
    {
        return new HtmlString(sprintf(
            '<div>
                <div class="metric-table-title">%s</div>
                <div class="metric-table-subtitle">%s</div>
            </div>',
            Address::states()[$row->ship_state] ?? $row->ship_state,
            $this->formatPercentage($row)
        ));
    }

    protected function formatCount($row)
    {
        return new HtmlString(sprintf(
            '<span class="metric-table-count pull-right">%s</span></a>',
            $row->amount
        ));
    }

    protected function formatPercentage($row)
    {
        return new HtmlString(sprintf(
            '<span style="display: inline-block; min-width: 70px; text-align: left;">%s%%</span>',
            $this->calculatePercentage($row->amount, $row->total)
        ));
    }
}
