/*
 * Menu Manager Plugin
 *
 * @Module Catalog
 * @Section Menus
 */
;( function( $, window, document, undefined ) {

  "use strict";

  var pluginName = "menuManager",
    defaults = {
      token: null,
      menuId: null,
      itemsField: '#items',
      loadUrl: "",
      addUrl: "",
      saveUrl: "",
      activeClass: 'jqtree-inactive',
      options: {
        selectable: true,
        autoOpen: true,
        dragAndDrop: true,
      },
      props: {
        active: '#node-active'
      },
      category: {
        button: '#node-add-action',
        selector: '#node-add-category'
      }
    };

  // The actual plugin constructor
  function Plugin (element, options) {
    this.element = element;
    this.$element = $(element);
    this.settings = $.extend({}, defaults, options);
    this._defaults = defaults;
    this._name = pluginName;
    this.init();
  }

  // Avoid Plugin.prototype conflicts
  $.extend(Plugin.prototype, {
    init: function() {
      var options = $.extend({
        dataUrl: this.settings.loadUrl,
        onDragStop: $.proxy(this.onDragStop, this),
        onCreateLi: $.proxy(this.onCreateLi, this),
        onCanMoveTo: $.proxy(this.onCanMoveTo, this)
      }, this.settings.options);
      this.$element.tree(options).on('tree.select', $.proxy(this.onTreeSelect, this));
      var _this = this;

      $(this.settings.props.active).on('change', function () {
        _this.toggleActive($(this).attr('data-id'), this.checked);
        _this.save();
      });

      $(this.settings.category.button).on('click', function (e) {
        var $categories = $(_this.settings.category.selector);
        var categoryId = parseInt($categories.val(), 10);
        var $selected = $categories.find('option[value=' + categoryId + ']');
        var categoryName = $selected.text();

        _this.add(categoryId, categoryName);
        $categories.find('option[value=' + categoryId + ']').remove();
      });

    },
    /**
     * The listener to call when the drag ends.
     *
     * @param {object} node
     * @param {Event} evt
     */
    onDragStop: function (node, evt) {
      this.save();
    },
    /**
     * The listener to call when jqTree creates a menu item.
     *
     * @param {object} node
     * @param {jQuery|HTMLElement} $li
     * @param {boolean} is_selected
     */
    onCreateLi: function(node, $li, is_selected) {
      $li.addClass(node.active ? '' : this.settings.activeClass);
    },
    /**
     * The listener to call when jqTree tries to determine if a selected item can move to another position.
     *
     * @param {object} moved_node
     * @param {object} target_node
     * @param {number} position
     * @returns {boolean}
     */
    onCanMoveTo: function(moved_node, target_node, position) {
      return (position !== 'inside');
    },
    /**
     * The listener to call when an item in jqTree is selected.
     *
     * @param {Event} event
     */
    onTreeSelect: function(event) {
      var form = $('#node-form');
      var formName = $('#node-name');
      var formActive = $('#node-active');

      if (event.node) {
        formName.text(event.node.name);
        formActive.val(event.node.active);
        formActive.attr('data-id', event.node.id);
        formActive.prop('checked', event.node.active);
        form.removeClass('hide');
      } else {
        form.addClass('hide');
      }
    },
    /**
     * Toggle the active state of a node.
     *
     * @param {number} id
     * @param {boolean} checked
     */
    toggleActive: function (id, checked) {
      var node = this.get(id);
      $(node.element).toggleClass(this.settings.activeClass, !checked);
      this.$element.tree('updateNode', node, {active: checked});
    },
    /**
     * Get a menu node by id.
     *
     * @param {number} id
     * @returns {Node}
     */
    get: function(id) {
      return this.$element.tree('getNodeById', id);
    },
    /**
     * Save the current state of the menu.
     */
    save: function() {
      if (this.settings.itemsField) {
        $(this.settings.itemsField).val(this.$element.tree('toJson'));
      }
    },
    /**
     * Add a new node to the menu.
     *
     * @param {number} categoryId
     * @param {string} categoryName
     */
    add: function(categoryId, categoryName) {
      if (categoryId && categoryName) {
        this.$element.tree('appendNode', {
          name: categoryName,
          catalog_menu_id: this.settings.menuId,
          catalog_category_id: categoryId,
          active: 1,
          parent_id: 0,
          id: 'new-' + categoryId,
        });
        this.save();
      }
    }
  });

  $.fn[pluginName] = function(options) {
    return this.each(function() {
      if (!$.data(this, "plugin_" + pluginName)) {
        $.data(this, "plugin_" + pluginName, new Plugin(this, options));
      }
    });
  };

})(jQuery, window, document);
