<?php

namespace Ignite\Catalog\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Models\Grid\EloquentTable;
use Yajra\DataTables\DataTables;

class CatalogItemTable extends EloquentTable
{
    /**
     * @var VendorFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    /**
     * CatalogItemTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param CatalogItemFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        CatalogItemFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'action' => [
                'name' => 'actions',
                'title' => 'Actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions'
            ],
            'created_at' => [
                'title' => 'Created At',
                'name' => 'created_at',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
            'name' => [
                'title' => 'Name',
                'name' => 'name',
                'orderable' => true,
                'exportable' => true,
            ],
            'manufacturer' => [
                'title' => 'Manufacturer',
                'name' => 'manufacturer',
                'searchable' => false,
                'orderable' => true,
                'exportable' => true,
            ],
            'class' => [
                'title' => 'Classification',
                'name' => 'class',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor' => [
                'title' => 'Vendor',
                'name' => 'vendor',
                'searchable' => false,
                'orderable' => false,
                'exportable' => true,
            ],
            'points' => [
                'title' => 'Points',
                'name' => 'points',
                'orderable' => true,
                'exportable' => true,
            ],
            'active' => [
                'title' => 'Active',
                'name' => 'active',
                'orderable' => false,
                'exportable' => true,
                'searchable' => false,
            ],
        ];
    }

    /**
     * The catalog items query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        $query = Item::with('vendor');

        $query->where('catalog_id', $this->catalog->getKey());

        return $query;
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'action' => [$this->formatter, 'actions'],
            'vendor' => [$this->formatter, 'vendor'],
            'active' => [$this->formatter, 'active'],
            'created_at' => [$this->formatter, 'createdAt'],
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        return parent::getBuilderParameters([
            "order" => [1, 'asc'] // Created At
        ]);
    }
}
