<?php

namespace Ignite\Catalog\Jobs;

use Exception;
use Ignite\Catalog\Contracts\VendorProcessor;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Entities\Vendor;
use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Support\Collection;
use RuntimeException;

class InteractWithVendor implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @var Order
     */
    protected $order;

    /**
     * Create a new job instance.
     *
     * @param  Order $order
     * @return void
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Execute the job.
     *
     * @return void
     * @throws Exception
     */
    public function handle()
    {
        try {
            $orderItems = $this->getOrderItems($this->order->getKey());

            /** @var Collection $orderItemsByVendorId */
            $orderItemsByVendorId = $orderItems->groupBy('catalog_vendor_id');

            /**
             * @var int $vendorId
             * @var Collection $items
             */
            foreach ($orderItemsByVendorId as $vendorId => $items) {
                try {
                    $processor = $this->resolveVendorProcessor($vendorId);
                } catch (Exception $exception) {
                    // If the execution reaches this point, then the vendor either doesn't exist or is not currently
                    // active. The most likely reason for this is an old order item getting processed after the
                    // vendor was deleted or disabled. In this case, the configuration of the Ignite program
                    // is not sync'd with the data in the database. We should probably just log the error.
                    logger()->debug($message = "Unable to resolve vendor factory for ID $vendorId.", [
                        'exception' => $exception
                    ]);
                    throw new RuntimeException($message, 0, $exception);
                }

                try {
                    $processor->process($this->order, $items->groupBy('class'));
                } catch (Exception $exception) {
                    logger()->debug($message = "Unable to process {$this->order->number}.", [
                        'exception' => $exception
                    ]);
                    throw new RuntimeException($message, 0, $exception);
                }
            }
        } catch (Exception $exception) {
            logger()->debug($exception->getMessage(), compact('exception'));
            $this->delete();
        }
    }

    /**
     * Get the collection of order items with their relationships.
     *
     * @param  int $orderId
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    private function getOrderItems($orderId)
    {
        return OrderItem::with('item', 'vendor')->where('catalog_order_id', $orderId)->get();
    }

    /**
     * Resolve the vendor factory class.
     *
     * @param  int $vendorId
     * @return VendorProcessor
     */
    private function resolveVendorProcessor($vendorId)
    {
        return Vendor::onlyActive()->get()
            ->keyBy('id')
            ->where('id', $vendorId)
            ->first()
            ->resolve();
    }
}
