<?php

namespace Ignite\Catalog\Http\Controllers;

use Exception;
use Ignite\Catalog\Collections\BuyRequestCollection;
use Ignite\Catalog\Events\AddItemToCartRequest;
use Ignite\Catalog\Events\AddItemToCartRequestComplete;
use Ignite\Catalog\Events\UpdateItemsInCartRequest;
use Ignite\Flash\Facades\Flash;
use Illuminate\Http\Request;
use Illuminate\Support\HtmlString;
use Ignite\Catalog\Contracts\CartRepository;
use Ignite\Catalog\Http\Requests\CartItemUpdateRequest;
use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Core\Http\Controllers\Controller;

class CartController extends Controller
{
    /**
     * @var CartRepository
     */
    private $cartRepository;

    /**
     * CartController constructor.
     *
     * @param CartRepository $cartRepository
     */
    public function __construct(CartRepository $cartRepository)
    {
        $this->cartRepository = $cartRepository;
    }

    /**
     * The index action lists the items in the cart.
     *
     * @param  Cart $cart
     * @return \Illuminate\View\View
     */
    public function index(Cart $cart)
    {
        return view('catalog.cart.index', compact('cart'));
    }

    /**
     * The store action adds an item to the user's cart.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            $buyRequest = BuyRequest::fromHttpRequest($request);

            event(new AddItemToCartRequest($buyRequest));

            if ($buyRequest->canProceed()) {
                $item = $this->cartRepository->add($buyRequest);

                Flash::success(__('Catalog::theme.cart.flash.add.success', [
                    'item' => $buyRequest->getItem()->present()->getName(),
                    'link' => route('catalog.cart.index')
                ]));

                event(new AddItemToCartRequestComplete($buyRequest, $item));
            }

            return redirect()->back();
        } catch (Exception $exception) {
            logger("Unable to add item to cart. Reason: {$exception->getMessage()}", compact('exception'));
            Flash::error(__('Catalog::theme.cart.flash.add.exception'));
            return redirect()->back();
        }
    }

    /**
     * The update action updates quantities of items in the cart.
     *
     * @param  CartItemUpdateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CartItemUpdateRequest $request)
    {
        $updates = $request->get('items', []);

        $buyRequests = (new BuyRequestCollection($updates))->map(function ($item, $itemId) {
            return new BuyRequest($itemId, $item['quantity'], ['class' => $item['class']]);
        });

        try {
            event(new UpdateItemsInCartRequest($buyRequests));

            if ($buyRequests->canProceed()) {
                $this->cartRepository->updateQuantities($buyRequests);
                Flash::success(__('Catalog::theme.cart.flash.update.success'));
            }
        } catch (Exception $exception) {
            logger($exception->getMessage(), compact('exception'));
            Flash::error(__('Catalog::theme.cart.flash.update.exception'));
        }

        return redirect()->route('catalog.cart.index');
    }

    /**
     * The destroy action removes an item from the user's cart.
     *
     * @param  Request $request
     * @param  int|null $id
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function destroy(Request $request, $id = null)
    {
        try {
            $cart = $this->cartRepository->remove($request->get('id', $id));
            $status = 'success';
            $message = __('Catalog::theme.cart.flash.remove.success', ['item' => $cart->item->name]);
        } catch (Exception $exception) {
            logger($exception->getMessage(), compact('exception'));
            $status = 'error';
            $message = __('Catalog::theme.cart.flash.remove.exception');
        }

        if ($request->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        Flash::add($message, $status);

        return redirect()->route('catalog.cart');
    }
}
