<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\MenuRepository;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Http\Forms\MenuForm;
use Ignite\Catalog\Models\Grid\MenusTable;
use Ignite\Core\Http\Controllers\Controller;
use Kris\LaravelFormBuilder\FormBuilder;

class MenusController extends Controller
{
    /**
     * Display all menus.
     *
     * @param  MenusTable $table
     * @param  Request    $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function index(MenusTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Catalog::admin.menus.index');
    }

    /**
     * Show a menu.
     *
     * @param  Menu $menu
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\Response
     */
    public function show(Menu $menu)
    {
        return view('Catalog::admin.menus.show', compact('menu'));
    }

    /**
     * Show the create form for a new menu.
     *
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(MenuForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.menus.create')
        ]);

        return view('Catalog::admin.menus.create', compact('form'));
    }

    /**
     * Store a new menu.
     *
     * @param  FormBuilder $formBuilder
     * @param  MenuRepository $repository
     * @return \Illuminate\Http\Response
     */
    public function store(FormBuilder $formBuilder, MenuRepository $repository)
    {
        $form = $formBuilder->create(MenuForm::class);

        if (! $form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $menu = $repository->create(collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::menu.store.success', ['name' => $menu->name]);
            return redirect()->route('admin.catalog.menus.edit', ['menu' => $menu->getKey()]);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::menu.store.failure');
            return redirect()->route('admin.catalog.menus.index');
        }
    }

    /**
     * Show the edit form for an existing menu.
     *
     * @param  Menu  $menu
     * @param  FormBuilder  $formBuilder
     * @return \Illuminate\Http\Response
     */
    public function edit(Menu $menu, FormBuilder $formBuilder)
    {
        $menu->load('items');

        $form = $formBuilder->create(MenuForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.menus.update', $menu),
            'model' => $menu
        ]);

        return view('Catalog::admin.menus.edit', compact('form', 'menu'));
    }

    /**
     * Update an existing attribute.
     *
     * @param  int $menu
     * @param  FormBuilder $formBuilder
     * @param  MenuRepository  $repository
     * @return \Illuminate\Http\Response
     */
    public function update($menu, FormBuilder $formBuilder, MenuRepository $repository)
    {
        $form = $formBuilder->create(MenuForm::class);

        if (!$form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        try {
            $menu = $repository->update($menu, collect($form->getFieldValues()));
            $this->flashSuccess('Catalog::menu.update.success', ['name' => $menu->name]);
            return redirect()->route('admin.catalog.menus.edit', $menu);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::menu.update.failure');
            return redirect()->route('admin.catalog.menus.index');
        }
    }

    /**
     * Delete one record.
     *
     * @param  int  $menu
     * @param  MenuRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($menu, MenuRepository $repository)
    {
        try {
            $menu = $repository->delete($menu);
            $status = 'success';
            $message = trans('Catalog::menu.delete.success', ['name' => $menu->name]);
        } catch (\Exception $e) {
            $this->logException($e);
            $status = 'error';
            $message = trans('Catalog::menu.delete.failure');
        }

        if (request()->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message
            ]);
        }

        $this->flash($status, $message);

        return redirect()->route('admin.catalog.menus.index');
    }

    /**
     * Delete multiple records at once.
     *
     * @param  MenuRepository  $repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroyMany(MenuRepository $repository)
    {
        $ids = request()->ids;
        $menus = explode(',', $ids);

        try {
            $repository->deleteMany($menus);
            $this->flashSuccess('Catalog::menu.delete_many.success', compact('ids'));
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('Catalog::menu.delete_many.failure');
        }

        return redirect()->route('admin.catalog.menus.index');
    }
}
