<?php

namespace Ignite\Catalog\Contracts;

use Illuminate\Support\Collection;

interface CatalogRepository
{
    /**
     * Find the catalog by code.
     *
     * @param  string $code
     * @return \Ignite\Catalog\Entities\Catalog
     */
    public function findByCode($code);

    /**
     * Find a catalog.
     *
     * @param  int    $id
     * @param  string $column
     * @return \Ignite\Catalog\Entities\Catalog
     */
    public function find($id, $column = 'id');

    /**
     * Find all catalogs.
     *
     * @return Collection
     */
    public function findAll();

    /**
     * Find all active catalogs.
     *
     * @return Collection
     */
    public function findAllActive();

    /**
     * Create a catalog.
     *
     * @param  Collection  $data
     * @return \Ignite\Catalog\Entities\Catalog
     */
    public function create(Collection $data);

    /**
     * Update a catalog.
     *
     * @paran  int  $id
     * @param  Collection  $data
     * @return \Ignite\Catalog\Entities\Catalog
     */
    public function update($id, Collection $data);

    /**
     * Paginate the catalogs.
     *
     * @param  int $perPage
     * @param  bool $activeOnly
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function paginate($perPage = 25, $activeOnly = true);

    /**
     * Search the catalog.
     *
     * @param  \Ignite\Catalog\Entities\Catalog $catalog
     * @param  string|null                      $keyword
     * @param  string|null                      $category
     * @param  string                           $sort
     * @param  int                              $perPage
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function search($catalog, $keyword = null, $category = null, $sort = 'name_asc', $perPage = 24);

    /**
     * Delete the catalog.
     *
     * @param  int $id
     * @return bool
     */
    public function delete(int $id);

    /**
     * Delete multiple records and any associated data.
     *
     * @param  array $ids
     * @return bool
     */
    public function deleteMany(array $ids);
}
