<?php

namespace Ignite\Catalog\Entities;

use Carbon\Carbon;
use Ignite\Catalog\Collections\OrderItemCollection;
use Ignite\Core\Facades\Format;
use Illuminate\Database\Eloquent\Model;

/**
 * Class OrderItem
 *
 * @property int $id
 * @property int $catalog_order_id
 * @property int $catalog_item_id
 * @property int $catalog_vendor_id
 * @property string $sku
 * @property string $name
 * @property string $type
 * @property string $class
 * @property int $quantity
 * @property int $cost
 * @property float $point_value
 * @property float $price
 * @property int $points
 * @property array $vendor_meta
 * @property bool $processed
 * @property int $processed_quantity
 * @property string $processed_at
 * @property bool $backordered
 * @property int $backordered_quantity
 * @property string $backordered_at
 * @property bool $shipped
 * @property int $shipped_quantity
 * @property string $shipped_at
 * @property bool $cancelled
 * @property int $cancelled_quantity
 * @property string $cancelled_at
 * @property \Ignite\Catalog\Entities\Order $order
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 * @property \Ignite\Catalog\Entities\Item $item
 */
class OrderItem extends Model
{
    /** @var string */
    protected $primaryKey = 'id';

    /** @var string */
    protected $table = 'catalog_order_item';

    /** @var array */
    protected $guarded = [];

    /** @var array */
    protected $casts = [
        'vendor_meta' => 'json'
    ];

    /**
     * The relationship to the catalog order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\Order::class, 'catalog_order_id', 'id');
    }

    /**
     * The relationship to the catalog item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function item()
    {
        return $this->hasOne(\Ignite\Catalog\Entities\Item::class, 'id', 'catalog_item_id');
    }

    /**
     * The relationship to the catalog item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\belongsTo
     */
    public function vendor()
    {
        return $this->belongsTo(\Ignite\Catalog\Entities\Vendor::class, 'catalog_vendor_id', 'id');
    }

    /**
     * The item Id.
     *
     * @return string
     */
    public function getItemId()
    {
        return $this->item->id;
    }

    /**
     * The image url for the item.
     *
     * @return string
     */
    public function getItemImageUrl()
    {
        return $this->item->getImageUrl();
    }

    /**
     * The item name.
     *
     * @return string
     */
    public function getItemName()
    {
        return $this->item->name;
    }

    /**
     * The item sku.
     *
     * @return string
     */
    public function getItemSku()
    {
        return $this->item->sku;
    }

    /**
     * The item quantity.
     *
     * @return string
     */
    public function getItemQuantity()
    {
        return number_format($this->quantity, 0);
    }

    /**
     * The item total point value.
     *
     * @return string
     */
    public function getItemPoints()
    {
        return Format::amount($this->points);
    }

    /**
     * The item total.
     *
     * @return string
     */
    public function getItemTotal()
    {
        return Format::amount($this->quantity * $this->points);
    }

    /**
     * Generate the request ID to send to the vendor.
     *
     * @return string
     */
    public function getVendorRequestId()
    {
        if (! $this->relationLoaded('order')) {
            $this->load('order');
        }

        return sprintf(
            '%s_%s_%s',
            $this->order->number,
            $this->getKey(),
            $this->created_at->format('Y_m_d_H_i_s')
        );
    }

    /**
     * Mark the item as processed.
     *
     * @param  array $values
     * @return OrderItem
     */
    public function process(array $values)
    {
        $quantity = (int) $values['quantity'] ?? $this->quantity;

        $this->processed_quantity = $quantity;
        $this->processed = 1;
        $this->processed_at = isset($values['date']) ? Carbon::parse($values['date']) : now();

        if ((int) $this->cancelled === 1) {
            $this->cancelled = 0;
            if ($quantity <= $this->cancelled_quantity) {
                $this->cancelled_quantity = $this->cancelled_quantity - $quantity;
            }
        }

        return $this;
    }

    /**
     * Mark the item as cancelled.
     *
     * @param  array $values
     * @return OrderItem
     */
    public function cancel(array $values)
    {
        $quantity = (int) $values['quantity'] ?? $this->quantity;

        $this->cancelled_quantity = $quantity;
        $this->cancelled = 1;
        $this->cancelled_at = isset($values['date']) ? Carbon::parse($values['date']) : now();

        if ((int) $this->processed === 1) {
            $this->processed = 0;
            if ($quantity <= $this->processed_quantity) {
                $this->processed_quantity = $this->processed_quantity - $quantity;
            }
        }

        return $this;
    }

    /**
     * Add an attempt record.
     *
     * @param  array $attempt
     * @return $this
     */
    public function addAttempt(array $attempt)
    {
        $this->setupAttempts();

        $attempts = $this->vendor_meta['attempts'];
        $attempts[$this->getVendorRequestId()] = $attempt;

        $this->vendor_meta = ['attempts' => $attempts];

        return $this;
    }

    /**
     * Ensure we have a place to store vendor attempts data.
     *
     * @return self
     */
    public function setupAttempts()
    {
        if (
            ! is_array($this->vendor_meta) ||
            ! isset($this->vendor_meta['attempts']) ||
            ! is_array($this->vendor_meta['attempts'])
        ) {
            $this->vendor_meta = ['attempts' => []];
        }

        return $this;
    }

    /**
     * Determine whether the order item is processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return (! $this->processed && ! $this->shipped && ! $this->cancelled && ! $this->backordered);
    }

    /**
     * Determine whether the order item has been processed.
     *
     * @return bool
     */
    public function isProcessed()
    {
        return (bool) $this->processed;
    }

    /**
     * Determine whether the order item has been shipped.
     *
     * @return bool
     */
    public function isShipped()
    {
        return (bool) $this->shipped;
    }

    /**
     * Determine whether the order item has been cancelled.
     *
     * @return bool
     */
    public function isCancelled()
    {
        return (bool) $this->cancelled;
    }

    /**
     * Determine whether the order item has been backordered.
     *
     * @return bool
     */
    public function isBackordered()
    {
        return (bool) $this->backordered;
    }

    /**
     * Determine if the order item can be processed.
     *
     * @return bool
     */
    public function isProcessable()
    {
        return (! $this->cancelled && ! $this->shipped);
    }

    /**
     * Determine if the order item can be processed.
     *
     * @return bool
     */
    public function isCancelable()
    {
        return (! $this->shipped);
    }

    /**
     * Determine if the order item can be processed.
     *
     * @return bool
     */
    public function isShippable()
    {
        return (! $this->isCancelled() && $this->isProcessed());
    }

    /**
     * Determine if the vendor is active for this item.
     *
     * @return bool
     */
    public function isVendorActive()
    {
        return (bool) $this->item->vendor_active;
    }

    /**
     * Count the number of vendor attempts.
     *
     * @return int
     */
    public function countVendorAttempts()
    {
        return (int) count($this->vendor_meta['attempts'] ?? []);
    }

    /**
     * Get the information from the first vendor attempt.
     *
     * @return array
     */
    public function getFirstVendorAttempt()
    {
        $attempts = collect($this->vendor_meta['attempts'] ?? []);

        return $attempts->first();
    }

    /**
     * Get the information from the last vendor attempt.
     *
     * @return array
     */
    public function getLastVendorAttempt()
    {
        $attempts = collect($this->vendor_meta['attempts'] ?? []);

        return $attempts->last();
    }

    /**
     * Determine if the attempt failed.
     *
     * @param array $attempt
     * @return bool
     */
    public function isFailedAttempt(array $attempt)
    {
        if (array_key_exists('errors', $attempt)) {
            return true;
        }

        if (array_key_exists('response', $attempt) && array_key_exists('errors', $attempt['response'])) {
            return true;
        }

        return false;
    }

    /**
     * New order items collection.
     *
     * @param array $models
     *
     * @return OrderItemCollection|\Illuminate\Database\Eloquent\Collection
     */
    public function newCollection(array $models = [])
    {
        return new OrderItemCollection($models);
    }
}
