<?php

namespace Ignite\Catalog\Models\Grid;

use Ignite\Core\Facades\Format;
use Illuminate\Support\HtmlString;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Core\Models\Grid\Formatter;

class OrderItemFormatter extends Formatter
{

    /**
     * Custom column for actions.
     *
     * @param  OrderItem $item
     * @return HtmlString
     */
    public function actions(OrderItem $item)
    {
        $actions = [];
        $actions[] = [
            'click' => '',
            'class' => 'open-item-inspector',
            'icon' => 'info-circle',
            'label' => 'Inspect',
            'text' => 'text-default',
            'type' => 'default',
            'link' => route('admin.catalog.orders.item.inspect', [
                'orderItem' => $item->getKey()
            ])
        ];

        if ($item->isProcessable()) {
            $actions[] = [
                'icon' => 'check',
                'type' => 'default',
                'text' => 'text-success',
                'class' => 'mark-item-modal',
                'label' => 'Mark Processed',
                'link' => route('admin.catalog.orders.item.status.edit', [
                    'orderItem' => $item->getKey(),
                    'status' => 'processed'
                ])
            ];
            $actions[] = [
                'icon' => 'ban',
                'text' => 'text-danger',
                'type' => 'default',
                'class' => 'mark-item-modal',
                'label' => 'Mark Cancelled',
                'link' => route('admin.catalog.orders.item.status.edit', [
                    'orderItem' => $item->getKey(),
                    'status' => 'cancelled'
                ])
            ];
        }

        /*[
            'icon' => 'truck',
            'label' => 'Ship',
            'text' => 'text-primary',
            'type' => 'default',
            'link' => route('admin.catalog.orders.item.mark', [
                'id' => $item->getKey(),
                'status' => 'shipped',
            ])
        ],
        [
            'icon' => 'arrow-left',
            'label' => 'Backorder',
            'text' => 'text-warning',
            'type' => 'default',
            'link' => route('admin.catalog.orders.item.mark', [
                'id' => $item->getKey(),
                'status' => 'backordered',
            ])
        ],*/

        return $this->html($this->buttons($actions));
    }

    /**
     * The order item image.
     *
     * @param  OrderItem $item
     * @return HtmlString
     */
    public function image(OrderItem $item)
    {
        return $this->html(sprintf(
            '<img src="%s" class="img-responsive" />',
            $item->getItemImageUrl()
        ));
    }

    /**
     * The order number.
     *
     * @param  OrderItem $item
     * @return HtmlString
     */
    public function number(OrderItem $item)
    {
        return $this->html($item->order->present()->getNumber());
    }

    /**
     * The order item processed state. Needs to also work in a report.
     *
     * @param  object $item
     * @return HtmlString
     */
    public function processed($item)
    {
        return $this->html($this->formatYesNo($item->processed));
    }

    /**
     * The order item points. Needs to also work in a report.
     *
     * @param  object $item
     * @return HtmlString
     */
    public function points($item)
    {
        return $this->html(Format::amount($item->points));
    }

    /**
     * The order item amount.
     *
     * @param  OrderItem $item
     * @return HtmlString
     */
    public function total($item)
    {
        if ($item instanceof OrderItem) {
            return $this->html($item->getItemTotal());
        }

        return Format::amount($item->total);
    }

    /**
     * The number of vendor interaction attempts.
     *
     * @param  OrderItem $item
     * @return int
     */
    public function attempts(OrderItem $item)
    {
        return $item->countVendorAttempts();
    }

    /**
     * Format whether an order has been processed.
     *
     * @param  OrderItem $item
     * @return HtmlString
     */
    public function status($item)
    {
        if ($item->isProcessing()) {
            $icon = $this->formatProcessingIcon();
        } elseif ($item->isProcessed()) {
            $icon = $this->formatProcessedIcon();
        } elseif ($item->isShipped()) {
            $icon = $this->formatShippedIcon();
        } elseif ($item->isCancelled()) {
            $icon = $this->formatCancelledIcon();
        } elseif ($item->isBackordered()) {
            $icon = $this->formatBackorderedIcon();
        } else {
            $icon = 'Unknown';
        }

        return new HtmlString($icon);
    }

    /**
     * Format a processing item.
     *
     * @return string
     */
    protected function formatProcessingIcon()
    {
        return sprintf('<span class="label label-warning">%1$s</span>', 'Processing');
    }

    /**
     * Format a processed item.
     *
     * @return string
     */
    protected function formatProcessedIcon()
    {
        return sprintf('<span class="label label-success">%1$s</span>', 'Processed');
    }

    /**
     * Format a shipped item.
     *
     * @return string
     */
    protected function formatShippedIcon()
    {
        return sprintf('<span class="label label-info">%1$s</span>', 'Shipped');
    }

    /**
     * Format a cancelled item.
     *
     * @return string
     */
    protected function formatCancelledIcon()
    {
        return sprintf('<span class="label label-danger">%1$s</span>', 'Cancelled');
    }

    /**
     * Format a backordered icon.
     *
     * @return string
     */
    protected function formatBackorderedIcon()
    {
        return sprintf('<span class="label label-default">%1$s</span>', 'Backordered');
    }
}
