<?php

namespace Ignite\Catalog\Listeners;

use Carbon\CarbonInterval;
use GuzzleHttp\Exception\GuzzleException;
use Ignite\Catalog\Events\IncompleteHawkOrder;
use Ignite\Catalog\Jobs\Hawk\ProcessManager;
use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\Response\BaseOrderResponse;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\Response;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\TransactionResponse;
use Ignite\Vendor\Hawk\ResponseException;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;

class RequestHawkOrderInfo implements ShouldQueue
{
    use InteractsWithQueue;

    /**
     * The time (seconds) before the job should be processed. E.g. 300 = 5 minutes.
     *
     * @var int
     */
    public $delay = 300;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ProcessManager
     */
    protected $manager;

    /**
     * RequestHawkOrderInfo constructor.
     *
     * @param Client $client
     */
    public function __construct(Client $client)
    {
        $this->client = $client;
    }

    /**
     * Handle the event.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    public function handle(IncompleteHawkOrder $event)
    {
        if ($event->response instanceof SubmitEGiftIndividualResponse && $event->orderItem->item->isEgift()) {
            return $this->handleEgiftIndividual($event);
        }

        if ($event->response instanceof SubmitClosedLoopIndividualResponse && $event->orderItem->item->isGiftcard()) {
            return $this->handleGiftcardIndividual($event);
        }

        if ($event->response instanceof SubmitOpenLoopPersonalizedIndividualResponse && $event->orderItem->item->isPrepaid()) {
            return $this->handlePrepaidIndividual($event);
        }

        if ($event->orderItem->item->isReloadable()) {
            if ($event->response instanceof SubmitOpenLoopPersonalizedIndividualResponse) {
                return $this->handleReloadableIndividual($event);
            }
            if ($event->response instanceof SubmitFundingResponse) {
                return $this->handleReloadableFunding($event);
            }
        }

        $this->delete();

        return false;
    }

    /**
     * Handle a SubmitEGiftIndividualResponse.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    protected function handleEgiftIndividual(IncompleteHawkOrder $event)
    {
        $manager = $this->getProcessManager($event, $requestId = $this->buildRequestId($event));

        /** @var SubmitEGiftIndividualTransactionResponse $response */
        $response = $this->client->submitEgiftIndividualTransaction(
            $this->getTransactionId($event),
            $requestId
        );

        // The order completed. Update the order item.
        if ($response instanceof SubmitEGiftIndividualTransactionResponse && $response->isCompleted()) {
            return $manager->complete($response->getResponse());
        }

        return $this->handleGenericResponse($response, $event, $requestId, $manager);
    }

    /**
     * Handle a SubmitClosedLoopIndividualResponse.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    protected function handleGiftcardIndividual(IncompleteHawkOrder $event)
    {
        $manager = $this->getProcessManager($event, $requestId = $this->buildRequestId($event));

        /** @var SubmitClosedLoopIndividualResponse $response */
        $response = $this->client->submitClosedLoopIndividualTransaction(
            $this->getTransactionId($event),
            $requestId
        );

        // The order completed. Update the order item.
        if ($response instanceof SubmitClosedLoopIndividualTransactionResponse && $response->isCompleted()) {
            return $manager->complete($response->getResponse());
        }

        return $this->handleGenericResponse($response, $event, $requestId, $manager);
    }

    /**
     * Handle a SubmitOpenLoopPersonalizedIndividualResponse.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    protected function handlePrepaidIndividual(IncompleteHawkOrder $event)
    {
        $manager = $this->getProcessManager($event, $requestId = $this->buildRequestId($event));

        /** @var SubmitOpenLoopPersonalizedIndividualTransactionResponse $response */
        $response = $this->client->submitOpenLoopPersonalizedIndividualTransaction(
            $this->getTransactionId($event),
            $requestId
        );

        // The order completed. Update the order item.
        if ($response instanceof SubmitOpenLoopPersonalizedIndividualTransactionResponse && $response->isCompleted()) {
            return $manager->complete($response->getResponse());
        }

        return $this->handleGenericResponse($response, $event, $requestId, $manager);
    }

    /**
     * Handle a reloadable SubmitOpenLoopPersonalizedIndividualResponse.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    protected function handleReloadableIndividual(IncompleteHawkOrder $event)
    {
        $manager = $this->getProcessManager($event, $requestId = $this->buildRequestId($event));

        /** @var SubmitOpenLoopPersonalizedIndividualTransactionResponse $response */
        $response = $this->client->submitOpenLoopPersonalizedIndividualTransaction(
            $this->getTransactionId($event),
            $requestId
        );

        // The order completed. Update the order item.
        if ($response instanceof SubmitClosedLoopIndividualTransactionResponse && $response->isCompleted()) {
            return $manager->complete($response->getResponse());
        }

        return $this->handleGenericResponse($response, $event, $requestId, $manager);
    }

    /**
     * Handle a reloadable SubmitFundingResponse.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return bool
     * @throws GuzzleException
     * @throws ResponseException
     */
    protected function handleReloadableFunding(IncompleteHawkOrder $event)
    {
        $manager = $this->getProcessManager($event, $requestId = $this->buildRequestId($event));

        /** @var SubmitFundingTransactionResponse $response */
        $response = $this->client->submitFundingTransaction(
            $this->getTransactionId($event),
            $requestId
        );

        // The order completed. Update the order item.
        if ($response instanceof SubmitFundingTransactionResponse && $response->isCompleted()) {
            return $manager->complete($response->getResponse());
        }

        return $this->handleGenericResponse($response, $event, $requestId, $manager);
    }

    /**
     * Handle the generic aspects of the response such as retry logic and error handling.
     *
     * @param Response $response
     * @param IncompleteHawkOrder $event
     * @param string $requestId
     * @param ProcessManager $manager
     *
     * @return bool
     * @throws \Exception
     */
    protected function handleGenericResponse(
        Response $response,
        IncompleteHawkOrder $event,
        string $requestId,
        ProcessManager $manager
    ) {
        // The request failed. Log it and send an error email to ops.
        if ($response instanceof ErrorResponse) {
            return $manager->error($response);
        }

        if (! $response instanceof TransactionResponse) {
            throw new \Exception('Unexpected response object.');
        }

        // The order failed, Update the order item and send an error email to ops.
        if ($response->isFailed()) {
            return $manager->error(new ErrorResponse(['exception' => [
                'requestId' => $requestId,
                'payload' => $response->responseToJson(),
            ]]));
        }

        // Log the attempt in every case of processing or queued.
        $manager->attempt($response);

        // The order is still processing or queued. Try again in 1 hour.
        if ($this->isAttemptNumber(1, $response)) {
            $this->release($this->getSecondAttemptDelay());
            return true;
        }

        // The order is still processing or queued. Try again in 1 day.
        if ($this->isAttemptNumber(2, $response)) {
            $this->release($this->getThirdAttemptDelay());
            return true;
        }

        // We tried 3 times over a day long period and the order is still processing,
        // let's send an error email to ops to follow up with Hawk support.
        $manager->error(new ErrorResponse(['exception' => [
            'requestId' => $requestId,
            'payload' => $response->responseToJson(),
        ]]));

        // Delete the job
        $this->delete();

        return false;
    }

    /**
     * Build up the request id.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return string
     */
    protected function buildRequestId(IncompleteHawkOrder $event)
    {
        return sprintf(
            '%s_%s_%s',
            $event->order->number,
            $event->orderItem->id,
            now()->format('Y_m_d_H_i_s')
        );
    }

    /**
     * Get the transaction from the previous response.
     *
     * @param IncompleteHawkOrder $event
     *
     * @return string|null
     */
    protected function getTransactionId(IncompleteHawkOrder $event)
    {
        return $event->response->getTransactionId() ?? null;
    }

    /**
     * Create an instance of a process manager.
     *
     * @param IncompleteHawkOrder $event
     * @param string $requestId
     *
     * @return ProcessManager
     */
    protected function getProcessManager(IncompleteHawkOrder $event, $requestId)
    {
        if (is_null($this->manager)) {
            return new ProcessManager($event->order, $event->orderItem, $requestId);
        }

        return $this->manager;
    }

    /**
     * Determine if the current number of attempts matches the given attempt number.
     *
     * @param int $attemptNumber
     * @param BaseOrderResponse|ErrorResponse|Response $response
     *
     * @return bool
     */
    protected function isAttemptNumber($attemptNumber, $response)
    {
        return $response instanceof SubmitEGiftIndividualTransactionResponse
               && ($response->isProcessing() || $response->isQueued())
               && $this->attempts() === $attemptNumber;
    }

    /**
     * The amount of time to delay before processing the second attempt.
     *
     * @return float|int
     */
    protected function getSecondAttemptDelay()
    {
        return CarbonInterval::hours(1)->totalSeconds;
    }

    /**
     * The amount of time to delay before processing the third attempt.
     *
     * @return float|int
     */
    protected function getThirdAttemptDelay()
    {
        return CarbonInterval::hours(24)->totalSeconds;
    }
}
