<?php

namespace Ignite\Catalog\Tests\Unit\Models\Product;

use DomainException;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Catalog\Tests\TestCase;
use Mockery;

class BuyRequestTest extends TestCase
{
    use RefreshDatabase;

    public function setUp() : void
    {
        parent::setUp();
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_can_be_constructed_with_a_simple_item()
    {
        $user = $this->loginAsAdmin();

        $vendor = factory(Vendor::class)->create();
        $catalog = factory(Catalog::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
        ]);
        $item = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
        ]);

        $buyRequest = new BuyRequest($item->getKey(), 2);

        $this->assertInstanceOf(Item::class, $buyRequest->getItem());
        $this->assertSame(100.0000, $buyRequest->getPoints());
        $this->assertSame(2, $buyRequest->getQuantity());
        $this->assertSame([
            'user_id' => $user->getKey(),
            'catalog_id' => $catalog->getKey(),
            'item_id' => $item->getKey(),
            'super_id' => $item->getKey(),
            'catalog_vendor_id' => $vendor->getKey(),
            'quantity' => $buyRequest->getQuantity(),
            'points' => $buyRequest->getPoints(),
            'price' => $item->price
        ], $buyRequest->toArray());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_can_be_constructed_with_a_configurable_item()
    {
        $user = $this->loginAsAdmin();

        $vendor = factory(Vendor::class)->create();
        $catalog = factory(Catalog::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
        ]);
        $configurable = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'configurable'
        ]);
        $simple = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'simple'
        ]);

        $buyRequest = new BuyRequest($configurable->getKey(), 2, [
            'item_id' => $simple->getKey()
        ]);

        $this->assertInstanceOf(Item::class, $buyRequest->getItem());
        $this->assertSame(100.0000, $buyRequest->getPoints());
        $this->assertSame(2, $buyRequest->getQuantity());
        $this->assertSame([
            'user_id' => $user->getKey(),
            'catalog_id' => $catalog->getKey(),
            'item_id' => $simple->getKey(),
            'super_id' => $configurable->getKey(),
            'catalog_vendor_id' => $vendor->getKey(),
            'quantity' => $buyRequest->getQuantity(),
            'points' => $buyRequest->getPoints(),
            'price' => $simple->price
        ], $buyRequest->toArray());
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_throws_an_exception_when_constructed_with_a_simple_item_and_options()
    {
        $this->expectException(DomainException::class);

        $vendor = factory(Vendor::class)->create();
        $catalog = factory(Catalog::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
        ]);
        $configurable = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'configurable'
        ]);

        new BuyRequest($configurable->getKey(), 2, []);
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_can_create_an_instance_from_a_http_request_object()
    {
        $vendor = factory(Vendor::class)->create();
        $catalog = factory(Catalog::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
        ]);
        $configurable = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'configurable'
        ]);
        $simple = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'simple'
        ]);

        $options = [
            'item_id' => $simple->getKey()
        ];

        $request = Mockery::mock('Illuminate\Http\Request');

        $request->shouldReceive('has')->with(BuyRequest::PARAM_ITEM_ID)->andReturn(true);
        $request->shouldReceive('get')->with(BuyRequest::PARAM_ITEM_ID)->andReturn($configurable->getKey());

        $request->shouldReceive('has')->with(BuyRequest::PARAM_QUANTITY)->andReturn(true);
        $request->shouldReceive('get')->with(BuyRequest::PARAM_QUANTITY)->andReturn(2);

        $request->shouldReceive('has')->with(BuyRequest::PARAM_OPTIONS)->andReturn(true);
        $request->shouldReceive('get')->with(BuyRequest::PARAM_OPTIONS, [])->andReturn($options);

        $this->assertInstanceOf(BuyRequest::class, BuyRequest::fromHttpRequest($request));
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_will_throw_an_exception_when_creating_an_instance_from_a_http_request_object_without_an_item_id()
    {
        $this->expectException(DomainException::class);

        $request = Mockery::mock('Illuminate\Http\Request');

        $request->shouldReceive('has')->with(BuyRequest::PARAM_ITEM_ID)->andReturn(false);

        BuyRequest::fromHttpRequest($request);
    }

    /**
     * @test
     * @group Catalog
     * @group Cart
     * @group BuyRequest
     */
    public function it_will_throw_an_exception_when_creating_an_instance_from_a_http_request_object_with_an_item_id_but_without_a_quantity()
    {
        $this->expectException(DomainException::class);

        $vendor = factory(Vendor::class)->create();
        $catalog = factory(Catalog::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
        ]);
        $configurable = factory(Item::class)->create([
            'catalog_vendor_id' => $vendor->getKey(),
            'catalog_id' => $catalog->getKey(),
            'points' => 100,
            'type' => 'configurable'
        ]);

        $request = Mockery::mock('Illuminate\Http\Request');

        $request->shouldReceive('has')->with(BuyRequest::PARAM_ITEM_ID)->andReturn(true);
        $request->shouldReceive('get')->with(BuyRequest::PARAM_ITEM_ID)->andReturn($configurable->getKey());
        $request->shouldReceive('has')->with(BuyRequest::PARAM_QUANTITY)->andReturn(false);

        BuyRequest::fromHttpRequest($request);
    }
}
