<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\Client as HawkApi;
use Ignite\Vendor\Hawk\FundingOrderLine;
use Ignite\Vendor\Hawk\PersonalizedOrderLine;
use Ignite\Vendor\Hawk\PersonalizedRecipient;
use Ignite\Vendor\Hawk\Recipient;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\ResponseException;
use Ignite\Vendor\Hawk\ShippingMethod;
use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

abstract class Reloadable extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $this->createCardOrReloadCard($payload, $requestId, $hawkApi);
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        if ($this->hasCard()) {
            return $this->buildSubmitFundingRequest();
        }

        return $this->buildOpenLoopPersonalizedIndividualRequest();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return [
            SubmitOpenLoopPersonalizedIndividualResponse::class,
            SubmitFundingResponse::class
        ];
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitOpenLoopPersonalizedIndividualRequest
     */
    protected function buildOpenLoopPersonalizedIndividualRequest()
    {
        return SubmitOpenLoopPersonalizedIndividualRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines(),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return PersonalizedOrderLine
     */
    protected function buildOrderLine()
    {
        return PersonalizedOrderLine::fromAssociativeArray([
            'amount' => (int) $this->orderItem->points,
            'description' => config('catalog.vendors.hawk.reloadable.description', ''),
            'fourthLineEmbossText' => config('catalog.vendors.hawk.reloadable.fourth_line_emboss', 'THANK YOU'),
            'cardCarrierMessage' => config('catalog.vendors.hawk.reloadable.card_carrier_message', ''),
            'shippingMethod' => config('catalog.vendors.hawk.reloadable.shipping_method', ShippingMethod::USPS_STANDARD_MAIL),
            'recipient' => $this->getRecipientData(),
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return PersonalizedRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return PersonalizedRecipient::fromAssociativeArray([
            'id' => $this->getId(),
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->getCountry(),
            ])
        ]);
    }

    /**
     * Create a card or reload it.
     *
     * @param $payload
     * @param $requestId
     * @param HawkApi $hawkApi
     *
     * @return ErrorResponse|SubmitFundingResponse|SubmitOpenLoopPersonalizedIndividualResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function createCardOrReloadCard($payload, $requestId, HawkApi $hawkApi)
    {
        if ($this->hasCard()) {
            return $hawkApi->submitFunding($payload, $requestId);
        }

        return $hawkApi->submitOpenLoopPersonalizedIndividual($payload, $requestId);
    }

    /**
     * Build the submit funding request.
     *
     * @return SubmitFundingRequest
     */
    protected function buildSubmitFundingRequest()
    {
        return SubmitFundingRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => [$this->buildFundingLines()],
        ]);
    }

    /**
     * Build the FundingOrderLine.
     *
     * @return FundingOrderLine
     */
    protected function buildFundingLines()
    {
        return FundingOrderLine::fromAssociativeArray([
            'amount' => (int) $this->orderItem->cost,
            'recipient' => Recipient::fromAssociativeArray([
                'id' => $this->getId(),
            ])
        ]);
    }

    /**
     * Determine whether the participant has a reloadable card already registered.
     *
     * @return bool
     */
    abstract protected function hasCard();

    /**
     * Get the participant identifier.
     *
     * @return string
     */
    abstract protected function getId();
}
