<?php

namespace Ignite\Catalog\Models;

use Ignite\Core\Facades\Format;
use Illuminate\Support\Collection;
use Ignite\Catalog\Entities\CartItem;

class Cart implements \Countable
{
    /**
     * @var Collection
     */
    private $items;

    /**
     * @var int
     */
    private $balance;

    /**
     * @var int
     */
    private $total;

    /**
     * Cart constructor.
     *
     * @param Collection $items
     * @param float      $balance
     * @param float      $total
     */
    public function __construct(Collection $items, float $balance, float $total)
    {
        $this->items = $items;
        $this->balance = $balance;
        $this->total = $total;
    }

    /**
     * The cart items.
     *
     * @return Collection
     */
    public function items()
    {
        return $this->items;
    }

    /**
     * The number of items in the cart.
     *
     * @return int
     */
    public function count()
    {
        return $this->quantity();
    }

    /**
     * Determine if the cart is empty.
     *
     * @return bool
     */
    public function empty()
    {
        return $this->items->isEmpty();
    }

    /**
     * Determine if the cart is not empty.
     *
     * @return bool
     */
    public function notEmpty()
    {
        return $this->items->isNotEmpty();
    }

    /**
     * The formatted total points of the items in the cart.
     *
     * @return string
     */
    public function total()
    {
        $total = Format::amount($this->items->sum(function (CartItem $item) {
            return $item->quantity * $item->points;
        }));

        $theTotal = Format::amount($this->total);

        if ($total != $theTotal) {
            throw new \DomainException('Calculated total in cart is different to the specified total');
        }

        return $theTotal;
    }

    /**
     * The total quantity of items in the cart.
     *
     * @return int
     */
    public function quantity()
    {
        return (int) $this->items->sum(function (CartItem $item) {
            return $item->getItemQuantity();
        });
    }

    /**
     * The user's total balance.
     *
     * @return string
     */
    public function balance()
    {
        return Format::amount($this->balance);
    }

    /**
     * The formatted deficit once the total items in the cart is subtracted from the user's balance.
     *
     * @return string
     */
    public function deficit()
    {
        $deficit = $this->balance - $this->total;

        return Format::amount($deficit);
    }

    /**
     * Determine whether the cart can be checked out.
     *
     * @return bool
     */
    public function canCheckout()
    {
        return ($this->balance - $this->total) >= 0;
    }

    /**
     * Determine whether the cart has a card item.
     *
     * @return bool
     */
    public function hasGiftCard()
    {
        return $this->items()->contains(function (CartItem $cartItem) {
            return $cartItem->isCard();
        });
    }

    /**
     * Determine whether the cart has at least one physical item.
     *
     * @return bool
     */
    public function hasPhysical()
    {
        return $this->countPhysical() > 0;
    }

    /**
     * Determine whether the cart contains only physical items.
     *
     * @return bool
     */
    public function hasOnlyPhysical()
    {
        return $this->items->count() === $this->countPhysical();
    }

    /**
     * The number of physical items in the cart.
     *
     * @return int
     */
    public function countPhysical()
    {
        return $this->items->filter(function (CartItem $model) {
            return $model->item->getClassInstance()->isPhysical();
        })->count();
    }

    /**
     * Determine whether the cart has at least one non-physical item.
     *
     * @return bool
     */
    public function hasNonPhysical()
    {
        return $this->countNonPhysical() > 0;
    }

    /**
     * Determine whether the cart contains only non-physical items.
     *
     * @return bool
     */
    public function hasOnlyNonPhysical()
    {
        return $this->items->count() === $this->countNonPhysical();
    }

    /**
     * The number of non-physical items in the cart.
     *
     * @return int
     */
    public function countNonPhysical()
    {
        return $this->items->filter(function (CartItem $model) {
            return ! $model->item->getClassInstance()->isPhysical();
        })->count();
    }
}
