<?php

namespace Ignite\Catalog\Console;

use Ignite\Core\Facades\Format;
use Illuminate\Console\Command;
use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Option;
use Ignite\Catalog\Entities\Vendor;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class Hawk extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:catalog:hawk';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import Hawk catalog data into Ignite';

    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $dryRunPositiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * The collection of categories.
     *
     * @var \Illuminate\Support\Collection
     */
    protected $categories;

    /**
     * The catalog to attach the items.
     *
     * @var Catalog
     */
    protected $catalog;

    /**
     * The catalog vendor to attach the items.
     *
     * @var Vendor
     */
    protected $vendor;

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $product = $this->argument('product');
        $filename = $this->option('file');
        $denominations = $this->option('denominations');

        $data = $this->getJsonData($filename);

        if (empty($data)) {
            throw new \DomainException("No data available in $filename.");
        }

        $this->process($data, $product, $denominations);

        return 0;
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->dryRunPositiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            ['product', InputArgument::REQUIRED, 'The product ID for this program.'],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['denominations', null, InputOption::VALUE_OPTIONAL, 'The denominations to create for each card.', null],
            ['point', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['markup', null, InputOption::VALUE_OPTIONAL, 'The value of a single point.', null],
            ['file', null, InputOption::VALUE_OPTIONAL, 'The file from which to import card data.', realpath(__DIR__ . "/data/hawk/giftcard.json")],
            ['dry-run', null, InputOption::VALUE_OPTIONAL, 'Should we run the import or dry-run it?', false]
        ];
    }

    /**
     * Handle an exception depending on the command verbosity.
     *
     * @param $e
     */
    protected function handleException($e)
    {
        if ($this->output->isVerbose() || $this->output->isVeryVerbose() || $this->output->isDebug()) {
            throw $e;
        }

        $this->error(
            sprintf('An error occurred: %s in %s on line %s', $e->getMessage(), $e->getFile(), $e->getLine())
        );
    }

    /**
     * The data from the JSON file.
     *
     * @param  string $filename
     * @return array
     */
    private function getJsonData($filename)
    {
        if (! starts_with($filename, '/')) {
            $filename = realpath(__DIR__ . "/data/hawk/$filename");
        }

        $json = file_get_contents($filename);

        $data = json_decode($json, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException("Unable to parse json in $filename. Error: " . json_last_error_msg());
        }

        if (! isset($data['data']) || ! is_array($data['data'])) {
            throw new \DomainException("Unable to find expected top-level key named 'data'.");
        }

        return $data['data'];
    }

    /**
     * Process the items.
     *
     * @param  array $data
     * @param  int   $product
     * @param  array $denominations
     * @return bool
     */
    private function process($data, $product, $denominations)
    {
        $rows = collect($data);
        $categories = collect();

        $this->info(sprintf('Importing %s %s.', $count = $rows->count(), str_plural('card', $count)));

        // Create the 'denomination' attribute if it does not already exist.
        $denominationAttribute = $this->getOrCreateDenominationAttribute();

        // Extrapolate the source data into Catalog Item entities.
        foreach ($rows as $number => $row) {
            if (isset($row['categories']) && ! empty($row['categories'])) {
                $this->categories(
                    $categories = $this->updateOrCreateCategories($row['categories'])->keyBy('code')
                );
                unset($row['categories']);
            }

            $row['code'] = str_slug($row['sku'], '-');

            // Create new Catalog Item instances for denominations.
            $simples = collect();
            $denominations = $row['denominations'] ?? $denominations;
            if (is_string($denominations)) {
                $denominations = explode(',', $denominations);
            }
            if (is_array($denominations)) {
                foreach ($denominations as $denomination) {
                    // Calculate points based on markup/margin.
                    $simples->push($this->createItemFromDenomination($denomination, $row, $product));
                }
            }

            $item = $this->createItem($row, $product);

            foreach ($simples as $index => $simple) {
                // Associate the simple items to the configurable as catalog_item_options.
                $item->options()->save(
                    $this->createOption($item, $simple, $denominationAttribute, $index)
                );
                // Associate the simple items to the configurable as catalog_attribute_items.
                $item->attributes()->associate(
                    $this->createAttributeItem($simple, $denominationAttribute, $index)
                );
            }

            // Associate both the simple and configurable items to the correct categories.
            $item->categories()->detach($categories);
            foreach ($categories as $category) {
                $item->categories()->attach($category->getKey(), [
                    'position' => $number + 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            unset($item->item_id);

            $item->points_min = $simples->min('points');
            $item->points_max = $simples->max('points');

            $item->save();
        }

        $this->info('Done.');

        return true;
    }

    /**
     * Ensure we have a denomination attribute.
     *
     * @return Attribute
     */
    private function getOrCreateDenominationAttribute()
    {
        return Attribute::updateOrCreate(['code' => 'denomination'], [
            'label' => 'Denomination',
            'code' => 'denomination',
            'type' => 'select'
        ]);
    }

    /**
     * Update or create categories.
     *
     * @param  string $categories
     * @return \Illuminate\Support\Collection
     */
    private function updateOrCreateCategories($categories)
    {
        return collect(explode('|', $categories))->map(function ($category) {
            $code = $this->getCategoryCode($category);
            return Category::updateOrCreate(['code' => $code], [
                'code' => $code,
                'name' => $category,
                'active' => 1
            ]);
        });
    }

    /**
     * Get the category code from the category name.
     *
     * @param  string $category
     * @return string
     */
    private function getCategoryCode($category)
    {
        return str_slug(str_replace(['&', '+'], 'and', $category));
    }

    /**
     * Get or merge the categories.
     *
     * @param  null|\Illuminate\Support\Collection $items
     * @return static
     */
    private function categories($items = null)
    {
        if (is_null($this->categories)) {
            $this->categories = collect();
        }

        if (! is_null($items)) {
            foreach ($items as $item) {
                $this->categories->push($item);
            }
        }

        return $this->categories->keyBy('code');
    }

    /**
     * Create an item from a denomination value.
     *
     * @param  int $denomination
     * @param  array $row
     * @param  int $product
     * @return Item
     */
    private function createItemFromDenomination($denomination, $row, $product)
    {
        $markup = (float) $this->option('markup') ?? $row['price_markup'];
        $point = (float) $this->option('point') ?? $row['point_value'];
        $denomination = (float) $denomination;

        $priceMarkedUp = $denomination + ($denomination * $row['price_markup']);

        $row['msrp'] = $denomination;
        $row['cost'] = $denomination;
        $row['price'] = $denomination;
        $row['price_markup'] = $markup;
        $row['price_margin'] = ($priceMarkedUp - $denomination) / $priceMarkedUp;
        $row['point_value'] = $point;
        $row['points'] = $this->calculatePoints($denomination, $markup, $point);
        $row['sku'] = $row['sku'] . '_' . $denomination;
        $row['code'] = $row['code'] . '-' . $denomination;
        $row['type'] = "simple";
        $row['name'] = sprintf('%s $%s', $row['name'], $denomination);
        $row['description'] = '';
        $row['terms'] = '';
        $row['visibility'] = 0;

        return $this->createItem($row, $product);
    }

    /**
     * Create an item.
     *
     * @param  array  $row
     * @param  string $product
     * @return Item
     */
    private function createItem($row, $product)
    {
        if (isset($row['denominations'])) {
            unset($row['denominations']);
        }

        if (! isset($row['catalog_vendor_id'])) {
            $row['catalog_vendor_id'] = $this->vendor()->getKey();
        }

        if (! isset($row['catalog_id'])) {
            $row['catalog_id'] = $this->catalog()->getKey();
        }

        if (! isset($row['msrp'])) {
            $row['msrp'] = 0.0000;
        }

        if (! isset($row['price'])) {
            $row['price'] = 0.0000;
        }

        if (! isset($row['points'])) {
            $row['points'] = 0.0000;
        }

        if (! isset($row['vendor_meta'])) {
            $row['vendor_meta'] = [];
        }

        $row['vendor_meta']['product_id'] = $product;

        return Item::updateOrCreate(['sku' => $row['sku']], $row);
    }

    /**
     * Calculate the points from the price, markup and point value.
     *
     * @param  float $denomination
     * @param  float $markup
     * @param  float $point
     * @return float
     */
    private function calculatePoints($denomination, $markup, $point)
    {
        return round($denomination * (1 + $markup) / $point, 0, PHP_ROUND_HALF_UP);
    }

    /**
     * Create the option.
     *
     * @param  Item      $item
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     *
     * @return Option
     */
    private function createOption($item, $simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'super_id' => $item->getKey(),
            'item_id' => $simple->getKey(),
            'attribute_id' => $attribute->getKey(),
            'value' => $value,
        ];

        $data = array_merge($where, [
            'label' => '$' . $value,
            'position' => $index + 1
        ]);

        return Option::updateOrCreate($where, $data);
    }

    /**
     * Create the association to the attribute.
     *
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     * @return AttributeItem
     */
    private function createAttributeItem($simple, $attribute, $index)
    {
        $value = Format::amount($simple->price);

        $where = [
            'attribute_id' => $attribute->getKey(),
            'item_id' => $simple->getKey(),
            'value' => $value
        ];

        $data = [
            'position' => $index + 1,
            'active' => 1
        ];

        return AttributeItem::updateOrCreate($where, $data);
    }

    /**
     * The catalog entity.
     *
     * @return Catalog
     * @throws \Exception
     */
    private function catalog()
    {
        if (is_null($this->catalog)) {
            $this->catalog = Catalog::where('type', 'card')->first();
            if (is_null($this->catalog)) {
                throw new \Exception('No catalogs exist. Run `module:seed` first.');
            }
        }

        return $this->catalog;
    }

    /**
     * The catalog vendor entity.
     *
     * @return Vendor
     * @throws \Exception
     */
    private function vendor()
    {
        if (is_null($this->catalog)) {
            $this->vendor = Vendor::where('name', 'hawk')->first();
            if (is_null($this->vendor)) {
                throw new \Exception('No vendors exist. Run `module:seed` first.');
            }
        }

        return $this->vendor;
    }
}
