<?php

namespace Ignite\Catalog\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Illuminate\Database\Eloquent\Builder;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Models\Grid\EloquentTable;
use Yajra\DataTables\DataTables;

class ItemsTable extends EloquentTable
{
    /**
     * @var VendorFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    /**
     * OrdersTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param array $params
     * @param ItemFormatter|null $formatter
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        array $params = [],
        ItemFormatter $formatter = null
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title'          => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            'action' => [
                'name' => 'actions',
                'title' => trans('Core::core.common.actions'),
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions'
            ],
            'created_at' => [
                'title' => trans('Catalog::item.fields.created_at.label'),
                'name' => 'catalog_item.created_at',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
            'image' => [
                'title' => trans('Catalog::item.fields.image.label'),
                'name' => 'catalog_item.image',
                'orderable' => true,
                'exportable' => true,
                'width' => '50px',
            ],
            'id' => [
                'title' => trans('Catalog::item.fields.id.label'),
                'name' => 'catalog_item.id',
                'orderable' => true,
                'exportable' => true,
            ],
            'name' => [
                'title' => trans('Catalog::item.fields.name.label'),
                'name' => 'catalog_item.name',
                'orderable' => true,
                'exportable' => true,
                'width' => '200px',
            ],
            'sku' => [
                'title' => trans('Catalog::item.fields.sku.label'),
                'name' => 'catalog_item.sku',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
            'type' => [
                'title' => trans('Catalog::item.fields.type.label'),
                'name' => 'catalog_item.type',
                'orderable' => true,
                'exportable' => true,
            ],
            'class' => [
                'title' => trans('Catalog::item.fields.class.label'),
                'name' => 'catalog_item.class',
                'orderable' => true,
                'exportable' => true,
            ],
            'manufacturer' => [
                'title' => trans('Catalog::item.fields.manufacturer.label'),
                'name' => 'catalog_item.manufacturer',
                'orderable' => true,
                'exportable' => true,
            ],
            'cost' => [
                'title' => trans('Catalog::item.fields.cost.label'),
                'name' => 'catalog_item.cost',
                'orderable' => true,
                'exportable' => true,
            ],
            'price' => [
                'title' => trans('Catalog::item.fields.price.label'),
                'name' => 'catalog_item.price',
                'orderable' => true,
                'exportable' => true,
            ],
            'price_markup' => [
                'title' => trans('Catalog::item.fields.price_markup.label'),
                'name' => 'catalog_item.price_markup',
                'orderable' => true,
                'exportable' => true,
            ],
            'price_margin' => [
                'title' => trans('Catalog::item.fields.price_margin.label'),
                'name' => 'catalog_item.price_margin',
                'orderable' => true,
                'exportable' => true,
            ],
            'points' => [
                'title' => trans('Catalog::item.fields.points.label'),
                'name' => 'catalog_item.points',
                'orderable' => true,
                'exportable' => true,
                'searchable' => false,
            ],
            'visibility' => [
                'title' => trans('Catalog::item.fields.visibility.label'),
                'name' => 'catalog_item.visibility',
                'orderable' => true,
                'exportable' => true,
            ],
            'active' => [
                'title' => trans('Catalog::item.fields.active.label'),
                'name' => 'catalog_item.active',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor' => [
                'title' => trans('Catalog::item.fields.catalog_vendor_id.label'),
                'name' => 'vendor.name',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor_active' => [
                'title' => trans('Catalog::item.fields.vendor_active.label'),
                'name' => 'catalog_item.vendor_active',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The report query.
     *
     * @return Builder
     */
    public function query()
    {
        $query = Item::with('catalog', 'vendor', 'categories');

        if (! is_null($this->visibility)) {
            $query->where('visibility', $this->visibility);
        }

        return $this->applyScopes($query);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'checkbox' => [$this->formatter, 'checkbox'],               /** @see ItemFormatter::checkbox()          */
            'action' => [$this->formatter, 'actions'],                  /** @see ItemFormatter::actions()           */
            'created_at' => [$this->formatter, 'createdAt'],            /** @see ItemFormatter::createdAt()         */
            'image' => [$this->formatter, 'image'],                     /** @see ItemFormatter::image()             */
            'vendor' => [$this->formatter, 'vendor'],                   /** @see ItemFormatter::vendor()            */
            'visibility' => [$this->formatter, 'visibility'],           /** @see ItemFormatter::visibility()        */
            'active' => [$this->formatter, 'active'],                   /** @see ItemFormatter::active()            */
            'vendor_active' => [$this->formatter, 'isVendorEnabled'],   /** @see ItemFormatter::isVendorEnabled()   */
            'price_margin' => [$this->formatter, 'margin'],             /** @see ItemFormatter::margin()            */
            'price_markup' => [$this->formatter, 'markup'],             /** @see ItemFormatter::markup()            */
            'price' => [$this->formatter, 'price'],                     /** @see ItemFormatter::price()             */
            'cost' => [$this->formatter, 'cost'],                       /** @see ItemFormatter::cost()              */
            'msrp' => [$this->formatter, 'msrp'],                       /** @see ItemFormatter::msrp()              */
            'points' => [$this->formatter, 'points'],                   /** @see ItemFormatter::points()            */
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        $params = parent::getBuilderParameters();

        return array_merge($params, [
            'order' => [2, 'asc'],
            'select' => [
                'info' => true,
                'style' => 'multi',
                'selector' => 'td:first-child'
            ],
            'columnDefs' => [
                ['targets' => 0, 'checkboxes' => ['selectRow' => true]]
            ],
            'buttons' => [
                'pageLength',
                $this->getColumnVisibilityButton(),
                'postCsv'
            ],
        ]);
    }
}
