<?php

namespace Ignite\Catalog\Entities;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

/**
 * Vendor
 *
 * @method static Builder onlyActive()
 * @property string $image_path
 */
class Vendor extends Model
{
    const ACTIVE = 1;
    const INACTIVE = 0;

    /** @var string */
    protected $primaryKey = 'id';

    /** @var string */
    protected $table = 'catalog_vendor';

    /** @var array */
    protected $guarded = [];

    /**
     * Set the vendor countries.
     *
     * @param array|string $countries
     */
    public function setVendorCountriesAttribute($countries)
    {
        if (is_array($countries)) {
            $countries = implode(',', array_map('trim', $countries));
        }

        $this->attributes['vendor_countries'] = $countries;
    }

    /**
     * Set the vendor countries.
     *
     * @param mixed $flag
     */
    public function setActiveAttribute($flag)
    {
        $this->attributes['active'] = (bool) $flag;
    }

    /**
     * Query for active vendors only.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeOnlyActive(Builder $query)
    {
        $query->where('active', self::ACTIVE);

        return $query;
    }

    /**
     * Query for multiple IDs.
     *
     * @param  Builder      $query
     * @param  array|string $ids
     * @return Builder
     */
    public function scopeByIds(Builder $query, $ids)
    {
        if (is_string($ids)) {
            $ids = preg_split('/\s?,\s?/', $ids);
        }

        $query->whereIn('id', $ids);

        return $query;
    }

    /**
     * Resolve the VendorFactory object for the specific vendor.
     *
     * @return \Ignite\Catalog\Contracts\VendorProcessor
     */
    public function resolve()
    {
        return app($this->redemption_class, ['vendor' => $this]);
    }

    /**
     * The presentation data for the model.
     *
     * @return array
     */
    public function toPresentableArray()
    {
        return [
            'active' => $this->active ? 'Yes' : 'No',
            'type' => ucwords($this->type),
            'vendor_countries' => $this->vendor_countries,
            'image_path' => $this->getVerifiedImagePathHtml(),
            'redemption_class' => $this->getVerifiedRedemptionClassHtml(),
        ];
    }

    /**
     * The html for the image path including existence check.
     *
     * @return string
     */
    private function getVerifiedImagePathHtml()
    {
        if (Storage::disk('ignite3')->exists($this->image_path)) {
            $html = '<sup class="text-success"><span class="fa fa-check"></span> path exists</sup>';
        } else {
            $html = '<sup class="text-danger"><span class="fa fa-times"></span> path does not exist</sup>';
        }

        return $this->image_path . ' ' .  $html;
    }

    /**
     * The html for the redemption class including existence check.
     *
     * @return string
     */
    private function getVerifiedRedemptionClassHtml()
    {
        if (class_exists($this->redemption_class)) {
            $html = '<sup class="text-success"><span class="fa fa-check"></span> class exists</sup>';
        } else {
            $html = '<sup class="text-danger"><span class="fa fa-times"></span> class does not exist</sup>';
        }

        return $this->redemption_class . ' ' . $html;
    }
}
