<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\PersonalizedOrderLine;
use Ignite\Vendor\Hawk\PersonalizedRecipient;
use Ignite\Vendor\Hawk\RecipientAddress;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\ShippingMethod;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;

class Prepaid extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitOpenLoopPersonalizedIndividual(
            $payload, $requestId
        );
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        return $this->buildOpenLoopPersonalizedIndividual();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponses()
    {
        return SubmitOpenLoopPersonalizedIndividualResponse::class;
    }

    /**
     * The OpenLoopPersonalizedIndividual data.
     *
     * @return SubmitOpenLoopPersonalizedIndividualRequest
     */
    protected function buildOpenLoopPersonalizedIndividual()
    {
        return SubmitOpenLoopPersonalizedIndividualRequest::fromAssociativeArray([
            'clientProgramNumber' => $this->getProductId(),
            'paymentType' => $this->getPaymentType(),
            'poNumber' => $this->order->number,
            'orderDetails' => $this->buildOrderLines(),
            "thirdPartyInfo" => [
                "address" => [
                    "company" => config('catalog.vendors.hawk.prepaid.company', ''),
                    "contact" => config('catalog.vendors.hawk.prepaid.contact', ''),
                    "line1" => config('catalog.vendors.hawk.prepaid.line1', ''),
                    "city" => config('catalog.vendors.hawk.prepaid.city', ''),
                    "region" => config('catalog.vendors.hawk.prepaid.region', ''),
                    "postalCode" => config('catalog.vendors.hawk.prepaid.postalCode', ''),
                    "country" => config('catalog.vendors.hawk.prepaid.country', ''),
                ],
                "taxId" => config('catalog.vendors.hawk.prepaidtax_id')
            ]
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return PersonalizedOrderLine
     */
    protected function buildOrderLine()
    {
        return PersonalizedOrderLine::fromAssociativeArray([
            'amount' => (int) $this->orderItem->cost,
            'description' => config('catalog.vendors.hawk.prepaid.description', ''),
            'fourthLineEmbossText' => config('catalog.vendors.hawk.prepaid.fourth_line_emboss', 'THANK YOU'),
            'cardCarrierMessage' => config('catalog.vendors.hawk.prepaid.card_carrier_message', ''),
            'shippingMethod' => config('catalog.vendors.hawk.prepaid.shipping_method', ShippingMethod::USPS_STANDARD_MAIL),
            'recipient' => $this->getRecipientData(),
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return PersonalizedRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return PersonalizedRecipient::fromAssociativeArray([
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'phoneNumber' => $this->order->ship_phone,
            'address' => RecipientAddress::fromAssociativeArray([
                'line1' => $this->order->ship_address_1,
                'line2' => $this->order->ship_address_2,
                'line3' => $this->order->ship_address_3,
                'city' => $this->order->ship_city,
                'region' => $this->order->ship_state,
                'postalCode' => $this->order->ship_postal,
                'country' => $this->getCountry(),
            ])
        ]);
    }

    /**
     * Get the product ID for the current environment.
     *
     * @return string
     */
    protected function getProductId()
    {
        if ('production' !== config('app.env')) {
            return config('catalog.vendors.hawk.test_prepaid_id', '43476908');
        } else {
            return $this->orderItem->item->vendor_meta['product_id'];
        }
    }
}
