<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Ignite\Vendor\Hawk\EGiftIndividualOrderLine;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\VirtualRecipient;
use Ignite\Vendor\Hawk\Client;

class Egift extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * @inheritdoc
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitEgiftIndividual($payload, $requestId);
    }

    /**
     * @inheritdoc
     */
    protected function payload()
    {
        return $this->buildEgiftIndividualRequest();
    }

    /**
     * @inheritdoc
     */
    protected function expectedResponse()
    {
        return SubmitEGiftIndividualResponse::class;
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitEGiftIndividualRequest
     */
    protected function buildEgiftIndividualRequest()
    {
        return SubmitEGiftIndividualRequest::fromAssociativeArray([
            'paymentType' => $this->getPaymentType(),
            'returnCardNumberAndPIN' => true,
            'poNumber' => $this->order->number,
            'clientProgramNumber' => $this->getProductId(),
            'orderDetails' => $this->buildOrderLines(),
            'emailContent' => app(EmailContent::class),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int) $this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return EgiftIndividualOrderLine
     */
    protected function buildOrderLine()
    {
        return EgiftIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int) $this->orderItem->cost,
            'contentProvider' => $this->orderItem->item->vendor_meta['content_provider_code'],
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return VirtualRecipient
     */
    protected function getRecipientData()
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);

        return VirtualRecipient::fromAssociativeArray([
            'firstName' => $first,
            'lastName' => $last,
            'email' => $this->order->ship_email,
            'id' => $this->order->user_id
        ]);
    }
}
