<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Illuminate\Http\Response as HttpResponse;
use Illuminate\Support\Facades\Mail;
use Ignite\Catalog\Emails\OrderItemFailure;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Vendor\Hawk\Response\BaseOrderResponse;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\Response;
use Illuminate\Support\Str;

abstract class Base
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * Create a new job instance.
     *
     * @param \Ignite\Catalog\Entities\Order     $order
     * @param \Ignite\Catalog\Entities\OrderItem $orderItem
     */
    public function __construct(Order $order, OrderItem $orderItem)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
    }

    /**
     * Execute the job.
     *
     * @param Client $hawkApi
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle(Client $hawkApi)
    {
        $requestId = $this->buildRequestId();
        $payload = $this->payload();
        $expectedResponse = $this->expectedResponse();

        try {
            $response = $this->request($payload, $requestId, $hawkApi);

            if ($response instanceof $expectedResponse) {
                if ($this->isComplete($response)) {
                    return $this->complete($response, $requestId);
                }
            }

            if ($response instanceof ErrorResponse) {
                return $this->error($response, $requestId);
            }

            return $this->attempt($response, $requestId, compact('payload'));
        } catch (\Exception $e) {
            return $this->error(new ErrorResponse(['exception' => [
                'requestId' => $requestId,
                'payload' => json_encode($payload->toArray(), JSON_PRETTY_PRINT),
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode(),
                'trace' => $e->getTraceAsString(),
            ]]), $requestId);
        }
    }

    /**
     * Send the request to hawk.
     *
     * @param $payload
     * @param string $requestId
     * @param Client $hawkApi
     *
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    abstract protected function request($payload, $requestId, Client $hawkApi);

    /**
     * The payload for the current request.
     *
     * @return SubmitOpenLoopPersonalizedIndividualRequest
     */
    abstract protected function payload();

    /**
     * The expected response for the current request.
     *
     * @return string
     */
    abstract protected function expectedResponse();

    /**
     * Build up the request id.
     *
     * @return string
     */
    protected function buildRequestId()
    {
        return sprintf(
            '%s_%s_%s',
            $this->order->number,
            $this->orderItem->id,
            $this->orderItem->created_at->format('Y_m_d_H_i_s')
        );
        //return $this->orderItem->getVendorRequestId()
    }

    /**
     * Complete the order.
     *
     * @param  BaseOrderResponse $response
     * @param  string            $requestId
     * @return bool
     */
    protected function complete(BaseOrderResponse $response, $requestId)
    {
        if ($response->getIsCompleted() && $response->getSuccess()) {
            $this->process($response, $requestId, [
                'processed' => 1,
                'processed_at' => now(),
                'processed_quantity' => $this->orderItem->quantity
            ]);
            return true;
        }

        return false;
    }

    /**
     * Determine whether the response is complete.
     *
     * @param BaseOrderResponse $response
     *
     * @return bool
     */
    protected function isComplete($response)
    {
        return $response->getStatusCode() === HttpResponse::HTTP_OK
               && $response->getIsCompleted()
               && $response->getSuccess();
    }

    /**
     * Store the attempt.
     *
     * @param BaseOrderResponse $response
     * @param string            $requestId
     * @param array             $meta
     *
     * @return bool
     */
    protected function attempt($response, $requestId, array $meta = [])
    {
        $orderItem = $this->process($response, $requestId, [], $meta);

        IncompleteOrder::dispatch($this->order, $orderItem, $response, $requestId);

        return true;
    }

    /**
     * Report the error.
     *
     * @param  ErrorResponse $response
     * @param  string $requestId
     * @return bool
     */
    protected function error(ErrorResponse $response, $requestId)
    {
        // We're not cancelling the item yet, instead, we'll send an email with the reason.
        $this->process($response, $requestId);

        Mail::send(new OrderItemFailure($this->order, $this->orderItem));

        return true;
    }

    /**
     * Process the response.
     *
     * @param  Response $response
     * @param  string   $requestId
     * @param  array    $data
     * @param  array    $meta
     * @return \Ignite\Catalog\Entities\OrderItem
     */
    protected function process(Response $response, $requestId, array $data = [], array $meta = [])
    {
        $existing = $this->orderItem->vendor_meta ?? [];

        if (! isset($existing['attempts'])) {
            $existing['attempts'] = [];
        }

        $existing['attempts'][$requestId] = [
            'response' => $response->toArray(),
            'item' => $this->orderItem->item->vendor_meta ?? []
        ];

        if (array_key_exists('attempts', $meta)) {
            unset($meta['attempts']);
        }

        $this->orderItem->update(array_merge([
            'vendor_meta' => array_merge($existing, $meta),
        ], $data));

        return $this->orderItem;
    }

    /**
     * The class key can be used to lookup config values.
     *
     * @return string
     */
    protected function getClassKey()
    {
        return Str::lower(Str::slug(class_basename(static::class), '_'));
    }

    /**
     * Determine the payment type to use.
     *
     * @return string
     */
    protected function getPaymentType()
    {
        if (is_array($this->orderItem->item->vendor_meta) &&
            isset($this->orderItem->item->vendor_meta['payment_type']) &&
            in_array($this->orderItem->item->vendor_meta['payment_type'], PaymentType::getPaymentTypes())
        ) {
            return $this->orderItem->item->vendor_meta['payment_type'];
        }

        $key = $this->getClassKey();

        return config("catalog.vendors.hawk.{$key}.payment_type", PaymentType::DRAW_DOWN);
    }

    /**
     * Get the product ID for the current environment.
     *
     * @return string
     */
    protected function getProductId()
    {
        if (app()->environment('production')) {
            return $this->orderItem->item->vendor_meta['product_id'];
        } else {
            $key = $this->getClassKey();
            return config("catalog.vendors.hawk.{$key}.test_product_id");
        }
    }

    /**
     * Format the country.
     *
     * @return string|null
     */
    protected function getCountry()
    {
        if (Str::upper($this->order->ship_country) === 'US') {
            return 'USA';
        }

        if (Str::upper($this->order->ship_country) === 'CA') {
            return 'CAN';
        }

        throw new \DomainException("Unsupported country: {$this->order->ship_country}");
    }
}
