<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Support\Collection;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Entities\MenuItem;

class MenuRepository implements \Ignite\Catalog\Contracts\MenuRepository
{
    /**
     * The default query builder.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Menu::query();
    }

    /**
     * Find a menu by id.
     *
     * @param  int  $id
     * @return Menu
     */
    public function find($id)
    {
        return Menu::findOrFail($id);
    }

    /**
     * Create a menu.
     *
     * @param  Collection  $data
     * @return Menu
     */
    public function create(Collection $data)
    {
        return Menu::create($data->except('items')->toArray());
    }

    /**
     * Update a menu.
     *
     * @param  int  $id
     * @param  Collection  $data
     * @return Menu
     */
    public function update(int $id, Collection $data)
    {
        $menu = $this->find($id);

        if ($data->has('items')) {
            $menu->syncItems($this->extractItems($data));
        }

        $menu->update($data->except('items')->toArray());

        return $menu;
    }

    /**
     * Extract the items data and map it into MenuItem objects.
     *
     * @param Collection  $data
     * @return array
     */
    protected function extractItems(Collection $data)
    {
        $models = [];
        $items = json_decode($data->pull('items', '[]'), true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException(
                'Expected a valid JSON string containing menu item data: ' . json_last_error_msg()
            );
        }

        foreach ($items as $position => $item) {
            $attributes = collect($item)->only(['catalog_category_id', 'catalog_menu_id', 'active', 'parent_id']);
            $model = new MenuItem($attributes->toArray());
            $model->position = $position + 1;
            $models[] = $model;
        }

        return $models;
    }

    /**
     * Delete a menu and any associated data.
     *
     * @param  int  $id
     * @return bool
     */
    public function delete(int $id)
    {
        return app('db')->transaction(function () use ($id) {
            $menu = $this->find($id);

            MenuItem::query()->where('catalog_menu_id', $id)->delete();

            return $menu->forceDelete();
        });
    }

    /**
     * Delete many menus and any associated data.
     *
     * @param  array  $ids
     * @return array
     */
    public function deleteMany(array $ids)
    {
        return app('db')->transaction(function () use ($ids) {
            MenuItem::query()->whereIn('catalog_menu_id', $ids)->delete();
            return Menu::query()->whereIn('id', $ids)->delete();
        });
    }
}
