<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Contracts\CartRepository as CartRepositoryContract;
use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Core\Entities\User;

class CartRepository implements CartRepositoryContract
{
    /**
     * The most basic cart query.
     *
     * @param  array $relations
     * @return \Illuminate\Database\Eloquent\Builder?
     */
    public function query($relations = [])
    {
        return CartItem::with(array_merge(['item', 'super'/*, 'catalog', 'vendor'*/], $relations));
    }

    /**
     * The query scoped for a user.
     *
     * @param  User|null $user
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function queryByUser(User $user = null)
    {
        $query = $this->query();

        if ($user instanceof User) {
            return $query->byUser($user);
        }

        return $query->byCurrentUser();
    }

    /**
     * Find cart items by a user. If a user is not provided, use the currently
     * logged in user. Optionally, pass whitelisted item ids to only return
     * those records.
     *
     * @param  User|null $user
     * @param  array     $whitelist
     * @return \Illuminate\Support\Collection
     */
    public function findByUser(User $user = null, array $whitelist = [])
    {
        $query = $this->queryByUser($user);

        if (! empty($whitelist)) {
            $query->forItems($whitelist);
        }

        return $query->get();
    }

    /**
     * Find all cart items. Not scoped by user.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Find a cart item.
     *
     * @param  int       $id
     * @param  string    $column
     * @param  User|null $user
     * @return \Illuminate\Database\Eloquent\Model|CartItem
     */
    public function find($id, $column = 'id', User $user = null)
    {
        return $this->queryByUser($user)->where($column, $id)->first();
    }

    /**
     * Find the total points in a user's cart. If the user is not specified, use the currently logged in user.
     *
     * @param  User|null $user
     * @return int
     */
    public function total(User $user = null)
    {
        return $this->queryByUser($user)->sum(DB::raw('quantity * points'));
    }

    /**
     * Add an item to the cart.
     *
     * @param  BuyRequest $buyRequest
     * @return CartItem
     */
    public function add(BuyRequest $buyRequest)
    {
        $col = 'item_id';
        $data = $buyRequest->toArray();
        $item = $buyRequest->getItem()->getKey();

        // If the item is already in the cart, just update the quantity.
        $existing = $this->find($item, $col);
        if ($existing) {
            $data['quantity'] = (int) $existing->quantity + $buyRequest->getQuantity();
        }

        return CartItem::updateOrCreate([$col => $item], $data);
    }

    /**
     * Remove an item from the cart.
     *
     * @param  int $id
     * @return \Illuminate\Database\Eloquent\Model|CartItem
     * @throws \Exception
     */
    public function remove($id)
    {
        $item = $this->find($id, 'item_id');

        if ($item) {
            $item->delete();
        }

        return $item;
    }

    /**
     * Update item quantities by passing along the array of items and quantity pairs:
     * e.g. $updates = [
     *      {item_id} => {quantity}
     * ];
     *
     * @param  array $updates
     * @return \Illuminate\Support\Collection
     */
    public function updateQuantities(array $updates)
    {
        $items = $this->findByUser(null, array_keys($updates));

        return $items->filter(function (CartItem $item) use ($updates) {
            return array_key_exists($item->item_id, $updates) || ! isset($updates[$item->item_id]['quantity']);
        })->each(function (CartItem $item) use ($updates) {
            $update = (int) $updates[$item->item_id]['quantity'];
            if ($update < 1) {
                return $item->delete();
            }
            return $item->update(['quantity' => $update]);
        });
    }
}
