<?php

namespace Ignite\Catalog\Http\Controllers;

use Illuminate\Http\Request;
use Ignite\Catalog\Contracts\FavoriteRepository;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Http\Controllers\Controller;

class FavoritesController extends Controller
{
    /**
     * @var FavoriteRepository
     */
    private $favoriteRepository;

    /**
     * FavoritesController constructor.
     *
     * @param FavoriteRepository $favoriteRepository
     */
    public function __construct(FavoriteRepository $favoriteRepository)
    {
        $this->favoriteRepository = $favoriteRepository;
    }

    /**
     * List all the users' favorited items.
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $items = $this->favoriteRepository->paginate(24);

        return view('catalog.favorites.index', compact('items'));
    }

    /**
     * Add the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Item $item)
    {
        try {
            $this->favoriteRepository->favorite($item);
            $message = 'Item successfully added to favorites.';
            $status = 'success';
        } catch (\Exception $e) {
            $message = 'Unable to favorite the item. Error: ' . $e->getMessage();
            $status = 'error';
        }

        flash()->message($message, $status);

        return redirect()->back();
    }

    /**
     * Destroy the relationship for the user to favorite the item.
     *
     * @param  Item $item
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Item $item)
    {
        try {
            $this->favoriteRepository->unfavorite($item);
            $message = 'Item successfully remove from favorites.';
            $status = 'success';
        } catch (\Exception $e) {
            $message = 'Unable remove the item from favorites. Error: ' . $e->getMessage();
            $status = 'error';
        }

        flash()->message($message, $status);

        return redirect()->back();
    }
}
